from __future__ import print_function

import argparse
import logging
import os
import re
import sys

import grpc
from google.protobuf import json_format
import prettytable

from infra.diskmanager.proto import diskman_pb2
from infra.diskmanager.lib import consts
from infra.diskmanager.lib.disk import Disk
from infra.diskmanager.lib.diskmanager_client import DiskManagerClient
from infra.diskmanager.lib import logger
import library.python.svn_version as sv

BYTES_IN_GB = 1024 * 1024 * 1024
log = logging.getLogger('dmctl')


class DiskmanClientException(Exception):
    pass


def size_suffix_str(value, pattern=re.compile('^([0-9]+)([KkMmGgTt]{0,1})$')):
    if not value:
        return value
    res = pattern.match(value)
    if not res:
        raise argparse.ArgumentTypeError('Invalid size value \"{}\", '
                                         'valid suffixes are Kk, Mm, Gg and Tt or none'.format(value))
    num_str, suffix = res.groups()
    numeric = int(num_str)
    if suffix.lower() == 'k':
        numeric *= 1024
    elif suffix.lower() == "m":
        numeric *= 1048576
    elif suffix.lower() == "g":
        numeric *= 1048576 * 1024
    elif suffix.lower() == "t":
        numeric *= 1048576 * 1048576
    return numeric


def kv_str(tok):
    if not tok:
        return tok
    try:
        k, v = tok.split('=', 1)
        if not v:
            raise argparse.ArgumentTypeError('Bad format, value can not be NULL')
    except ValueError:
        raise argparse.ArgumentTypeError('Bad kv format')
    return (k, v)


class version_action(argparse.Action):
    def __call__(self, parser, args, values, option_string=None):
        print(sv.svn_version())
        parser.exit()


def dump_disks_list(disks, out, header=''):
    t = prettytable.PrettyTable(
        ['Disk Id', 'Name', 'Class',  'UUID', 'Capacity', 'Allocatable', 'FSTRIM', 'Ready', 'Configured', 'Absent', 'Error'])
    t.align['Capacity'] = 'l'
    disks.sort(key=lambda d: d.spec.device_path)
    for d in disks:
        t.add_row([d.meta.id,
                   d.spec.device_path,
                   d.spec.storage_class,
                   d.spec.layout_id,
                   '{:0.3f} Gb'.format(float(d.spec.capacity_bytes) / BYTES_IN_GB),
                   '{:0.3f} Gb'.format(float(d.status.allocatable_bytes) / BYTES_IN_GB),
                   diskman_pb2.DiskSpec.FSTrimPolicy.Name(d.spec.fstrim),
                   d.status.ready.status,
                   d.status.configured.status,
                   d.status.absent.status,
                   d.status.error.status])

    if header:
        out.write(header)
        out.write('\n')
    out.write(t.get_string())
    out.write('\n')
    out.flush()


def dump_json_msg(msg, out, header=''):
    msg = json_format.MessageToJson(msg, preserving_proto_field_name=True, sort_keys=True,
                                    including_default_value_fields=True)
    out.write(msg)
    out.write('\n')


def list_disks(args, stub, out, header=''):
    req = diskman_pb2.ListDisksRequest()
    if hasattr(args, 'ids'):
        req.disk_ids.extend(args.ids)

    resp = stub.ListDisks(req)
    if args.format_json:
        dump_json_msg(resp, out, header)
    else:
        dump_disks_list(resp.disks, out, header)


def list_volumes(args, stub, out, header=''):
    req = diskman_pb2.ListVolumesRequest()
    if hasattr(args, 'ids'):
        req.volume_ids.extend(args.ids)
    if hasattr(args, 'mnt_paths'):
        req.mount_paths.extend(args.mnt_paths)

    resp = stub.ListVolumes(req)
    if args.format_json:
        dump_json_msg(resp, out, header)
    else:
        dump_volumes_list(resp.volumes, out, header)


def dump_volumes_list(volumes, out, header=''):
    t = prettytable.PrettyTable(
        ['Volume Id', 'Name', 'Disk ID', 'Capacity', 'Type', 'FS', 'MountOpt', 'FSTRIM', 'Mpath', 'Manageable',
         'Configured', 'Ready'])
    for v in volumes:
        if v.spec.HasField('mount'):
            fs_type = v.spec.mount.fs_type
            fstrim = v.spec.mount.fstrim
            fs_opt = diskman_pb2.VolumeSpec.MountVolume.MountPolicy.Name(v.spec.mount.mount_policy)
            if fs_opt == 'CUSTOM':
                fs_opt += ':' + v.spec.mount.mount_flags
        else:
            fs_type = ''
            fs_opt = ''
            fstrim = False
        t.add_row([v.meta.id,
                   v.spec.name,
                   v.spec.disk_id,
                   '{:0.3f} Gb'.format(float(v.spec.capacity_bytes) / BYTES_IN_GB),
                   v.spec.WhichOneof("access_type"),
                   fs_type,
                   fs_opt,
                   fstrim,
                   v.status.mount_path,
                   v.status.manageable.status,
                   v.status.configured.status,
                   v.status.ready.status])
    if header:
        out.write(header)
        out.write('\n')
    out.write(t.get_string())
    out.write('\n')
    out.flush()


def list_all(args, stub, out):
    list_disks(args, stub, out, header='Disks')
    list_volumes(args, stub, out, header='Volumes')


def part_parse(p_opts, part):
    #  [start,size,type,volume_source]
    try:
        start_str, size_str, p_type, p_vol = p_opts.split(',')
    except ValueError as e:
        raise argparse.ArgumentTypeError('Bad partition format %s' % str(e))

    start = size_suffix_str(start_str)
    size = size_suffix_str(size_str)
    part.start_bytes = start
    part.size_bytes = size
    part.type = p_type
    part.volume_source = bool(p_vol)


def generate_default_part(disk_spec):
    # One large GPT partition with lvm group, alignment 1Mb
    alignment = 1024 ** 2
    start = alignment
    end = (long(disk_spec.capacity_bytes) / alignment) * alignment
    ret = "{},{},{},{}".format(start, end - start, Disk.PART_TYPES['linux_lvm'], 1)
    return ret


def format_disk(args, stub, out):
    req = diskman_pb2.FormatDiskRequest()
    req.disk_id = args.id
    req.force = args.force
    req.fstrim = diskman_pb2.DiskSpec.FSTrimPolicy.Value(args.fstrim.upper())

    disk = None
    for d in stub.ListDisks(diskman_pb2.ListDisksRequest()).disks:
        if d.meta.id == req.disk_id:
            disk = d
            break
    if disk is None:
        raise DiskmanClientException('Format failed. Can not find disk: %s ' % req.disk_id)

    # If parition layout was not provides, use default one,
    if args.partition is None:
        def_part = generate_default_part(disk.spec)
        args.partition = [def_part]

    for p_str in args.partition:
        p = req.partitions.add()
        part_parse(p_str, p)

    for k, v in args.labels:
        req.labels[k] = v

    resp = stub.FormatDisk(req)
    if args.format_json:
        dump_json_msg(resp, out)
    else:
        dump_disks_list([resp.disk], out)


def create_volume(args, stub, out):
    req = diskman_pb2.CreateVolumeRequest()
    v_spec = req.volume_spec
    v_spec.name = args.name
    v_spec.disk_id = args.disk
    v_spec.capacity_bytes = args.size
    v_spec.readonly = args.readonly
    if args.type == 'mount':
        v_spec.mount.fs_type = args.fs_type
        v_spec.mount.root_owner.uid = args.owner_uid
        v_spec.mount.root_owner.gid = args.owner_gid
        v_spec.mount.mount_policy = diskman_pb2.VolumeSpec.MountVolume.MountPolicy.Value(args.mount_policy.upper())
        if v_spec.mount.mount_policy == diskman_pb2.VolumeSpec.MountVolume.MountPolicy.Value('CUSTOM'):
            v_spec.mount.mount_flags = args.mount_flags
        elif args.mount_flags:
            raise argparse.ArgumentTypeError('mount_opts is conflict with mount_policy %s, use "CUSTOM" policy instead' % args.mount_policy)

        v_spec.mount.fstrim = args.fstrim
    else:
        v_spec.block.stub = True

    for k, v in args.labels:
        v_spec.labels[k] = v

    resp = stub.CreateVolume(req)

    if args.format_json:
        dump_json_msg(resp, out)
    else:
        out.write(resp.volume.meta.id)
        out.write('\n')
        out.flush()


def delete_volume(args, stub, out):
    req = diskman_pb2.DeleteVolumeRequest()
    req.volume_id = args.id
    stub.DeleteVolume(req)


def mount_volume(args, stub, out):
    req = diskman_pb2.MountVolumeRequest()
    req.volume_id = args.id
    req.mount_path = os.path.realpath(args.mount_path)
    stub.MountVolume(req)


def umount_volume(args, stub, out):
    req = diskman_pb2.UmountVolumeRequest()
    req.volume_id = args.id
    stub.UmountVolume(req)


def set_iolimit_volume(args, stub, out):
    req = diskman_pb2.SetIOLimitRequest()
    req.id = args.id
    req.iolimit.read.ops_per_second = args.read_iops
    req.iolimit.read.bytes_per_second = args.read_bps
    req.iolimit.write.ops_per_second = args.write_iops
    req.iolimit.write.bytes_per_second = args.write_bps
    stub.SetIOLimitVolume(req)


def get_yt_devs(args, stub, out, header=''):
    resp = stub.GetYTMountedDevices(diskman_pb2.GetYTMountedDevicesRequest())
    dump_json_msg(resp, out, header)


def get_hot_swap_creds(args, stub, out, header=''):
    resp = stub.GetHotSwapCreds(diskman_pb2.GetHotSwapCredsRequest())
    dump_json_msg(resp, out, header)


def daemon_get_stat(args, stub, out, header=''):
    resp = stub.DaemonGetStat(diskman_pb2.DaemonGetStatRequest())
    dump_json_msg(resp, out, header)


def daemon_update_cache(args, stub, out, header=''):
    stub.DaemonUpdateCache(diskman_pb2.DaemonUpdateCacheRequest())


def daemon_set_loglevel(args, stub, out):
    req = diskman_pb2.DaemonSetLoglevelRequest()
    req.verbosity = args.verbosity
    stub.DaemonSetLoglevel(req)


def get_server_info(args, stub, out, header=''):
    resp = stub.ServerInfo(diskman_pb2.ServerInfoRequest())
    dump_json_msg(resp, out, header)


def init_argparse(parser):
    parser.add_argument("-S", "--service", dest="service",
                        type=str, default=consts.DEFAULT_SERVER_ADDR,
                        help='service address, default: %(default)s')
    parser.add_argument("-v", "--verbose", dest="verbose",
                        action="count", default=0,
                        help="increases log verbosity for each occurence.")
    parser.add_argument('-J', '--json', dest='format_json', default=False, action='store_true')
    parser.add_argument('-V', '--version', action=version_action, nargs=0)

    subparsers = parser.add_subparsers(title="Possible actions")

    srv_info = subparsers.add_parser(name="server-info", description='Get server info')
    srv_info.set_defaults(handle=get_server_info)

    ls = subparsers.add_parser(name="list", description='List all resources')
    ls.set_defaults(handle=list_all)
    ls.add_argument('--status', dest='show_status', default=False, action='store_true')
    ls.add_argument('-w', '--wide', dest='show_all', default=False, action='store_true')

    disk_ls = subparsers.add_parser(name="disk-list", description='List disks')
    disk_ls.set_defaults(handle=list_disks)
    disk_ls.add_argument('--ids', dest='ids', type=str, nargs='*', default=[], help='Disk ids (if no id is given - list all)')
    disk_ls.add_argument('--status', dest='show_status', default=False, action='store_true')
    disk_ls.add_argument('-w', '--wide', dest='show_all', default=False, action='store_true')

    vol_ls = subparsers.add_parser(name="vol-list", description='List volumes')
    vol_ls.set_defaults(handle=list_volumes)
    vol_ls.add_argument('--ids', dest='ids', type=str, nargs='*', default=[], help='Volume ids (if no id is given - list all)')
    vol_ls.add_argument('--mnt_paths', dest='mnt_paths', type=str, nargs='*', default=[], help='Volume mount paths (if no path is given - list all volumes)')
    vol_ls.add_argument('--status', dest='show_status', default=False, action='store_true')
    vol_ls.add_argument('-w', '--wide', dest='show_all', default=False, action='store_true')

    disk_format = subparsers.add_parser(name="disk-format", description='Format disk')
    disk_format.set_defaults(handle=format_disk)
    disk_format.add_argument('id', type=str, help='Disk id')
    disk_format.add_argument('--force', dest='force', default=False, action='store_true')
    disk_format.add_argument('--partition', type=str, action='append',
                             help="Partition in following format [start,size,type,volume_source]")
    disk_format.add_argument('fstrim', default='auto', const='auto', nargs='?',
                             choices=['auto', 'enabled', 'disabled'])
    disk_format.add_argument('--label', dest='labels', type=kv_str, action='append', default=[],
                             help="Add label to disk in key=value form")

    vol_create = subparsers.add_parser(name="vol-create", description='Create volume')
    vol_create.set_defaults(handle=create_volume)
    vol_create.add_argument('name', type=str, help="Volume name")
    vol_create.add_argument('disk', type=str, help='Source disk id')
    vol_create.add_argument('size', type=size_suffix_str, help="Desired volume size in bytes")
    vol_create.add_argument('type', default='mount', const='mount', nargs='?', choices=['mount', 'block'])
    vol_create.add_argument('-r', '--read-only', dest='readonly', default=False, action='store_true')
    vol_create.add_argument('--fs_type', dest='fs_type', type=str, default='ext4')
    vol_create.add_argument('--mount_policy', dest='mount_policy', default='default', choices=['default', 'safe', 'unsafe', 'rootfs', 'custom'])
    vol_create.add_argument('--mount_opts', dest='mount_flags', type=str, default='',
                            help='Comma separated filesystem options list, only for mount_policy=CUSTOM')
    vol_create.add_argument('--uid', dest='owner_uid', type=int, default=os.getuid(), help='Set root_owner owner user id')
    vol_create.add_argument('--gid', dest='owner_gid', type=int, default=consts.DEFAULT_GROUP_ID,
                            help='Set root_owner owner group id')
    vol_create.add_argument('-F', '--no-fstrim', dest='fstrim', default=True, action='store_false')
    vol_create.add_argument('--label', dest='labels', type=kv_str, action='append', default=[],
                             help="Add label to disk in key=value form")

    vol_delete = subparsers.add_parser(name="vol-delete", description='Delete volume')
    vol_delete.set_defaults(handle=delete_volume)
    vol_delete.add_argument('id', type=str, help='volume id')

    vol_mount = subparsers.add_parser(name="vol-mount", description='Mount volume')
    vol_mount.set_defaults(handle=mount_volume)
    vol_mount.add_argument('id', type=str, help='volume id')
    vol_mount.add_argument('mount_path', type=str, help='volume mount path')

    vol_umount = subparsers.add_parser(name="vol-umount", description='Umount volume')
    vol_umount.set_defaults(handle=umount_volume)
    vol_umount.add_argument('id', type=str, help='volume id')

    vol_setlim = subparsers.add_parser(name="vol-set-iolimit", description='Set iolimit for volume')
    vol_setlim.set_defaults(handle=set_iolimit_volume)
    vol_setlim.add_argument('id', type=str, help='volume id')
    vol_setlim.add_argument('read_iops', type=int, help='Read operations per second')
    vol_setlim.add_argument('read_bps', type=int, help='Read bytes per second')
    vol_setlim.add_argument('write_iops', type=int, help='Write operations per second')
    vol_setlim.add_argument('write_bps', type=int, help='Write bytes per second')

    yt_devs = subparsers.add_parser(name="get-yt-mounted-devices", description='Get paths and mountpoints of devices mounted inside /yt')
    yt_devs.set_defaults(handle=get_yt_devs)

    hot_swap_creds = subparsers.add_parser(name="get-hot-swap-creds", description='Get hot swap credentials')
    hot_swap_creds.set_defaults(handle=get_hot_swap_creds)

    daemon_stat = subparsers.add_parser(name="daemon-stat", description='Fetch daemon stats')
    daemon_stat.set_defaults(handle=daemon_get_stat)

    daemon_ucache = subparsers.add_parser(name="daemon-update-cache", description='Force daemon cache update')
    daemon_ucache.set_defaults(handle=daemon_update_cache)

    daemon_set_ll = subparsers.add_parser(name="daemon-set-loglevel", description='Set daemon loglevel')
    daemon_set_ll.set_defaults(handle=daemon_set_loglevel)
    daemon_set_ll.add_argument('verbosity', type=int, help='Set log level (0 - ERROR, 1 - WARNING, 2 - INFO, else - DEBUG)')


def main():
    parser = argparse.ArgumentParser(description='Diskmanager CLI tool (rev:%s)' % sv.svn_revision(),
                                     epilog='Documentation: %s' % consts.WIKI_URL)

    init_argparse(parser)
    if not len(sys.argv[1:]):
        args = parser.parse_args(['list'])
    else:
        args = parser.parse_args(sys.argv[1:])
    logger.setup_logger('diskmanager', args.verbose)
    try:
        with grpc.insecure_channel(args.service) as channel:
            stub = DiskManagerClient(channel)
            args.handle(args, stub, sys.stdout)
    except grpc.RpcError as e:
        print("gRPC request ({}) failed: code={}, details='{}'".format(args.service,
                                                                       e.code().name,
                                                                       e.details()))
        exit(1)
    except DiskmanClientException as e:
        print("Error: %s " % str(e))
        exit(1)
    except argparse.ArgumentTypeError as e:
        print("Bad option: %s " % str(e))
        exit(1)


if __name__ == '__main__':
    main()
