import os
import logging

log = logging.getLogger('diskmanager.dirutil')


def wrap_error(f, *args, **kwargs):
    try:
        log.debug("calling {}.{}, args: {}, kwargs: {}".format(f.__module__, f.__name__, args, kwargs))
        rv = f(*args, **kwargs)
        log.debug("return value: {}".format(rv))
        return rv, None
    except Exception as e:
        return None, 'got exception: {}'.format(e)


def ensure_dir(path, owner, group, mode, makedirs=True):
    if makedirs is True:
        mkdir = os.makedirs
    else:
        mkdir = os.mkdir

    if not os.path.isdir(path):
        _, err = wrap_error(mkdir, path, mode)
        if err:
            return 'failed to create directory {}: {}'.format(path, err)
        _, err = wrap_error(os.chown, path, owner, group)
        if err:
            return 'failed to chown directory {}: {}'.format(path, err)
    elif os.path.islink(path):
        return 'path {} is symlink'.format(path)
    elif os.path.isfile(path):
        return 'path {} is file'.format(path)
    else:
        s, err = wrap_error(os.stat, path)
        if err:
            return 'failed to stat directory {}: {}'.format(path, err)
        if s.st_uid != owner or s.st_gid != group:
            _, err = wrap_error(os.chown, path, owner, group)
            if err:
                return 'failed to ensure ownership for {}:{} on {}: {}'.format(owner, group, path, err)
        if s.st_mode & 0o777 != mode:
            _, err = wrap_error(os.chmod, path, mode)
            if err:
                return 'failed to ensure mode {} on {}: {}'.format(mode, path, err)
    return None


def ensure_link(path, target, makedirs=True):
    if not os.path.islink(path):
        parent_dir = os.path.dirname(path)
        if makedirs is True and not os.path.isdir(parent_dir):
            _, err = wrap_error(os.makedirs, parent_dir)
            if err:
                return 'failed to create parent dir {} for link {}: {}'.format(parent_dir, path, err)
        _, err = wrap_error(os.symlink, target, path)
        if err:
            return 'failed to create link {} to {}: {}'.format(path, target, err)
    elif os.path.islink(path):
        current_target = os.readlink(path)
        if current_target != target:
            tmp = '{}-tmp'.format(path)
            _, err = wrap_error(os.symlink, target, tmp)
            if err:
                return 'failed to create temporary link {} to {}: {}'.format(tmp, target, err)
            _, err = wrap_error(os.rename, tmp, path)
            if err:
                return 'failed to rename temporary link {} to {}: {}'.format(tmp, path, err)
    return None
