import consts
import logging
import json
import disk
from google.protobuf.json_format import ParseDict
from infra.diskmanager.proto import diskman_pb2

log = logging.getLogger('diskmanager.lib.limit')


class IOLimitInitError(Exception):
    pass


class IOLimitInfo(object):
    _SYS_BLKIO_THR_PREFIX = '/sys/fs/cgroup/blkio/blkio.throttle.'

    def __init__(self, devnum):
        self._dev = devnum
        self._read_iops = 0
        self._write_iops = 0
        self._read_bps = 0
        self._write_bps = 0
        self._read_sysfs()

    def _sysfs_file(self, name):
        return self._SYS_BLKIO_THR_PREFIX + name + '_device'

    def _read_sysfs_val(self, name):
        with open(self._sysfs_file(name), 'rb') as f:
            for line in f.readlines():
                tok = line.split(" ")
                if len(tok) != 2:
                    raise Exception('Unexpected data from sysfs %s, line: "%s"' % (name, line))
                if self._dev == tok[0]:
                    return int(tok[1])
        return 0

    def _set_sysfs_val(self, name, value):
        data = "%s %s" % (self._dev, value)
        disk.write_sysfs_file(data, self._sysfs_file(name))

    def _read_sysfs(self):
        self._read_iops = self._read_sysfs_val('read_iops')
        self._read_bps = self._read_sysfs_val('read_bps')
        self._write_iops = self._read_sysfs_val('write_iops')
        self._write_bps = self._read_sysfs_val('write_bps')

    def set_read_iops(self, iops):
        if iops and iops < consts.DEFAULT_IOLIM_READ_IOPS_MIN:
            raise IOLimitInitError("Read iops limit is too low: want:%d, minimal allowed:%s",
                                   iops, consts.DEFAULT_IOLIM_READ_IOPS_MIN)
        self._read_iops = iops
        self._set_sysfs_val('read_iops', iops)

    def set_write_iops(self, iops):
        if iops and iops < consts.DEFAULT_IOLIM_WRITE_IOPS_MIN:
            raise IOLimitInitError("Write iops limit is too low: want:%d, minimal allowed:%s",
                                   iops, consts.DEFAULT_IOLIM_WRITE_IOPS_MIN)
        self._write_iops = iops
        self._set_sysfs_val('write_iops', iops)

    def set_read_bps(self, bps):
        if bps and bps < consts.DEFAULT_IOLIM_READ_BPS_MIN:
            raise IOLimitInitError("Read bps limit is too low: want:%d, minimal allowed:%s",
                                   bps, consts.DEFAULT_IOLIM_READ_BPS_MIN)
        self._read_bps = bps
        self._set_sysfs_val('read_bps', bps)

    def set_write_bps(self, bps):
        if bps and bps < consts.DEFAULT_IOLIM_WRITE_BPS_MIN:
            raise IOLimitInitError("Write bps limit is too low: want:%d, minimal allowed:%s",
                                   bps, consts.DEFAULT_IOLIM_WRITE_BPS_MIN)
        self._write_bps = bps
        self._set_sysfs_val('write_bps', bps)


class IOLimitsForYPExport(object):

    def __init__(self, config_file='', cache=None, open_func=open):
        if cache is None:
            self._limits = {}
            self._config_file = config_file
        else:
            self._limits = cache._limits
            self._config_file = cache._config_file
        self.update(open_func=open_func)

    def update(self, open_func=open):
        filtered_limits = {}
        try:
            limits_config = open_func(self._config_file)
        except Exception as e:
            log.warning('Cannot open io-limits file %s: %s' % (self._config_file, str(e)))
            return

        with limits_config:
            try:
                all_limits = json.load(limits_config)
            except Exception as e:
                log.warning('Cannot load io-limits: %s' % str(e))
                limits_config.close()
                return

        for model in all_limits.keys():
            try:
                new_msg = diskman_pb2.YPIOLimits()
                ParseDict(all_limits[model], new_msg)
            except Exception as e:
                log.warning('disk model "%s" has invalid properties: %s' % (model, str(e)))
            else:
                new_msg.use_override = True
                filtered_limits[model] = new_msg

        if self._limits != filtered_limits:
            self._limits = filtered_limits
            log.info('io-limits for YP was updated')

    def get(self, _disk):
        # using serial as model be cause there is no ID_MODEL for nvme in udev
        model = _disk.serial if _disk.storage_class == consts.STORAGE_NVME else _disk.model
        for key in self._limits.keys():
            if model.find(key) > -1:
                return self._limits[key]
        return None
