import functools
import logging
import logging.handlers
import traceback

DEF_LOG_FORMAT = '%(asctime)s %(levelname)s %(module)s:%(lineno)d %(message)s'


def set_log_level(logger, verbose):
    if not verbose:
        logger.setLevel(logging.ERROR)
    elif verbose == 1:
        logger.setLevel(logging.WARNING)
    elif verbose == 2:
        logger.setLevel(logging.INFO)
    else:
        logger.setLevel(logging.DEBUG)


def setup_logger(name, verbose):
    logger = logging.getLogger(name)
    set_log_level(logger, verbose)
    ch = logging.StreamHandler()
    formatter = logging.Formatter(DEF_LOG_FORMAT)
    ch.setFormatter(formatter)
    logger.addHandler(ch)
    return logger


def grpc_client_id(context):
    metadata = dict(context.invocation_metadata())
    return metadata.get('client_id', 'unknown')


def get_req_id(request, context):
    return '%s/%s' % (grpc_client_id(context), id(request))


def logrpc(f=None, logger=None):
    if not logger:
        return f

    if not f:
        return functools.partial(logrpc, logger=logger)

    def tab(what):
        return '\t' + '\n\t'.join(filter(None, str(what).split('\n')))

    @functools.wraps(f)
    def dolog(self, request, context):
        req_id = get_req_id(request, context)
        if request.ListFields():
            msg = ' params\n%s' % tab(request)
        else:
            msg = 'out params'
        logger.debug('=> GRPC [%s]: %s with%s\n' %
                         (req_id, f.__name__, msg))
        try:
            result = f(self, request, context)
        except Exception as exc:
            if context._state.code:
                code = str(context._state.code)[11:]
                details = context._state.details
                tback = ''
            else:
                code = 'Unexpected exception'
                details = exc.message
                tback = '\n' + tab(traceback.format_exc())
            logger.debug('!! GRPC [%s]: %s on %s (%s)%s\n' %
                             (req_id, code, f.__name__, details, tback))
            raise
        if str(result):
            str_result = '\n%s' % tab(result)
        else:
            str_result = ' nothing'
        logger.debug('<= GRPC [%s]: %s returns%s\n' %
                         (req_id, f.__name__, str_result))

        return result
    return dolog


def log_add_file(logger, fname, maxBytes, backupCount=3):
    fh = logging.handlers.RotatingFileHandler(fname, maxBytes=maxBytes, backupCount=backupCount)
    fmt = logging.Formatter(DEF_LOG_FORMAT)
    fh.setFormatter(fmt)
    logger.addHandler(fh)
    return logger
