#!/usr/bin/env python
#
import subprocess
import shlex
import logging

# This class impement minimal subset of LVM2 distro agnostic helpers.
# Assumptions:
#  Call native lvm-cmds:
#   Native lvm's pythons binding (see [1]) is known to be broken for a long time
#   so the only reliable way is to  call native lvm cmds.
#  Manual parsing:
#   Unfortunately older lvm-tools has not json output support
#   even more lvmtools can not be compiled statically (see[2])
#   so we have to manually parse cmd output
# Links
# [1] https://github.com/lvmteam/lvm2/blob/master/python/example.py
# [2] https://github.com/moby/moby/issues/34865

LVM2_PREFIX = 'LVM2_'


class LVMException(Exception):
    pass


class LVM(object):
    _logger = logging.getLogger('diskmanager.lib.lvm')

    @staticmethod
    def _parse_kv_pipe(pipe):
        rc = []
        return rc

    @staticmethod
    def get_capacity(val):
        if val.endswith('B'):
            return int(val[:-1])
        return int(val)

    @staticmethod
    def _ls_cmd(cmd, root_helper=None):
        rc = []
        if root_helper:
            cmd = [root_helper] + cmd
        try:
            LVM._logger.debug("exec : %s" % cmd)
            p = subprocess.Popen(cmd, stdout=subprocess.PIPE)
            for line in iter(p.stdout.readline, ''):
                # Convert nameprefixed output to dict like follows:
                # LVM2_VG_NAME=vg_name... => { 'vg_name':'vg_name'...}
                entry = {}
                d = dict(token.split('=', 1) for token in shlex.split(line))
                for k in d:
                    if k.startswith(LVM2_PREFIX):
                        k_strip = k.replace(LVM2_PREFIX, '').lower()
                        entry[k_strip] = d[k]
                        # Convert XX_tags comma separated string to dict:
                        # key=val1,key2=val2 => {'key1':'val1'..}
                        if k_strip.endswith('_tags'):
                            tag_dict = {}
                            tag_lex = shlex.shlex(entry[k_strip])
                            tag_lex.whitespace_split = True
                            tag_lex.whitespace = ','
                            for token in tag_lex:
                                if token.find('=') == -1:
                                    k = token
                                    v = 'true'
                                else:
                                    k, v = token.split('=', 1)
                                tag_dict[k] = v
                            entry[k_strip] = tag_dict
                if len(entry):
                    rc.append(entry)
            code = p.wait()
            if code:
                raise Exception('cmd:%s fail with:%d' % (cmd, code))
        except OSError as e:
            # Do not try to hide error from ls methods, if something is wrong let's caller know it
            raise e
        return rc

    @staticmethod
    def _do_exec(cmd):
        LVM._logger.debug("exec : %s" % cmd)
        subprocess.check_call(cmd)

    @staticmethod
    def list_vg(name=None):
        cmd = ['vgs', '--noheadings', '--nameprefixes', '--units', 'b', '--options', 'vg_all,pv_name,pv_uuid']
        if name:
            cmd.append(name)
        return LVM._ls_cmd(cmd)

    @staticmethod
    def list_lv(name=None):
        cmd = ['lvs', '--noheadings', '--nameprefixes', '--units', 'b', '--options', 'lv_all,vg_name']
        if name:
            cmd.append(name)
        return LVM._ls_cmd(cmd)

    @staticmethod
    def list_pv(name=None):
        cmd = ['pvs', '--noheadings', '--nameprefixes', '--units', 'b', '--options', 'pv_all']
        if name:
            cmd.append(name)
        return LVM._ls_cmd(cmd)

    @staticmethod
    def create_vg(name, pv_list, tags=None):
        if tags is None:
            tags = []
        cmd = ['vgcreate', '-y', name] + pv_list
        for tag in tags:
            cmd.append('--addtag')
            cmd.append(tag)
        LVM._do_exec(cmd)

    @staticmethod
    def change_vg_tags(name, add_tags=None, del_tags=None):
        cmd = ['vgchange', '-y', name]
        if add_tags is None:
            add_tags = []
        if del_tags is None:
            del_tags = []
        for tag in add_tags:
            cmd.append('--addtag')
            cmd.append(tag)
        for tag in del_tags:
            cmd.append('--deltag')
            cmd.append(tag)
        LVM._do_exec(cmd)

    @staticmethod
    def delete_vg(name, force=False):
        cmd = ['vgremove', name]
        if force:
            cmd.append('-ff')
        LVM._do_exec(cmd)

    @staticmethod
    def create_pv(name, dev):
        cmd = ['pvcreate', name, dev]
        LVM._do_exec(cmd)

    @staticmethod
    def delete_pv(name, force=False):
        cmd = ['pvremove', name]
        if force:
            cmd.append('-ff')
        LVM._do_exec(cmd)

    @staticmethod
    def create_lv(name, size, vg_name, tags=None):
        if tags is None:
            tags = []
        cmd = ['lvcreate', '-y', '--wipesignatures', 'y', '--name', name, '--size', str(size) + 'B', vg_name]
        for tag in tags:
            cmd.append('--addtag')
            cmd.append(tag)
        LVM._do_exec(cmd)

    @staticmethod
    def change_lv_tags(name, vg_name, add_tags=None, del_tags=None):
        cmd = ['lvchange', '-y', vg_name + "/" + name]
        if add_tags is None:
            add_tags = []
        if del_tags is None:
            del_tags = []
        for tag in add_tags:
            cmd.append('--addtag')
            cmd.append(tag)
        for tag in del_tags:
            cmd.append('--deltag')
            cmd.append(tag)
        LVM._do_exec(cmd)

    @staticmethod
    def delete_lv(name, vg_name, force=False):
        cmd = ['lvremove', '-y', vg_name + "/" + name]
        if force:
            cmd.append('-ff')
        LVM._do_exec(cmd)
