import copy
import logging
import os
import subprocess

log = logging.getLogger('diskmanager.lib.mount')


def run(args, **kwargs):
    log.debug("exec cmd: '{}'".format(" ".join(args)))
    return subprocess.check_call(args, **kwargs)


class MountInfo(object):
    def __init__(self, idx, parent, dev, root, path, opt, fs_type, fs_dev, fs_opt):
        self.id = idx
        self.parent = parent
        self.dev = dev
        self.root = root
        self.path = path
        self.fs_type = fs_type
        self.fs_dev = fs_dev
        self.gen_opts = opt.split(',')
        self.fs_opts = fs_opt.split(',')
        self.opts = list(set(self.gen_opts + self.fs_opts))


def read_mountinfo():
    ret = []
    with open('/proc/self/mountinfo', 'rb') as f:
        for line in f:
            ts = line.strip().split(' ')
            if len(ts) < 10 or ts[-4] != '-':
                raise Exception('Unexpected data from procfs, line: "%s"' % line)
            ret.append(MountInfo(ts[0], ts[1], ts[2], ts[3], ts[4], ts[5], ts[-3], ts[-2], ts[-1]))
    return ret


def find_by_dev(major, minor, cache=None):
    if cache is None:
        cache = read_mountinfo()
    devno = major + ':' + minor
    for m in cache:
        if m.dev == devno and m.root == '/':
            return m
    return None


def find_by_path(path, cache=None):
    if cache is None:
        cache = read_mountinfo()
    for m in cache:
        if m.path == path and m.root == '/':
            return m
    return None


def find_mount_path(major, minor, cache=None):
    m = find_by_dev(major, minor, cache)
    if m is not None:
        return m.path
    return ""


def project_quota_on(mnt_path):
    if not os.path.exists(os.path.join(mnt_path, "quota.project")):
        run(["project_quota", "init", mnt_path])
    run(["project_quota", "on", mnt_path])


class MountOpt(object):
    def __init__(self):
        pass

    def is_eq(self, opt_list):
        return False

    # Returns (add_list,  del_list)
    def diff(self, opt_list):
        return ([], [])


class MountOptBin(object):
    def __init__(self, on, off, default, state):
        self.on = on
        self.off = off
        self.default = default
        self.state = state

    def __str__(self):
        return "MountOptBin(on:{}, off:{}, default:{}, state:{})".format(
            self.on, self.off, self.default, self.state)

    def is_eq(self, opt_list):
        state = self.default

        if self.off in opt_list:
            state = False
        elif self.on in opt_list:
            state = True
        return state == self.state

    def diff(self, opt_list):

        if self.is_eq(opt_list):
            return ([], [])

        if not self.state:
            return ([self.off], [self.on])
        return ([self.on], [self.off])


class MountConfig(object):
    def __init__(self, default, options, error, project_quota=False, mkfs_opts={'opts': [], 'ext_opts': []}, reserved_space=5):
        self.default = default
        self.options = options
        self.error = error
        self.project_quota = project_quota
        self.mkfs_opts = mkfs_opts
        self.reserved_space = reserved_space  # in percent

    def get_defaults(self):
        return ','.join(self.default)

    def is_healthy(self, opts):
        for o in opts:
            if o in self.error:
                return False
        return True

    def need_remount(self, opt_list):
        for o in self.options:
            if not o.is_eq(opt_list):
                log.debug("mount opt:{} not in: {}".format(o, opt_list))
                return True
        return False

    def make_opt(self, opt_list):
        if not opt_list:
            return self.default

        new = opt_list[:]
        on = []
        off = []
        for o in self.options:
            e, d = o.diff(opt_list)
            on += e
            off += d

        for o in off:
            if o in new:
                new.remove(o)
        new += on
        return new

    def get_mkfs_opts(self):
        return copy.deepcopy(self.mkfs_opts)
