#!/usr/bin/env python
#
import logging
import os
import subprocess
import uuid
from infra.diskmanager.lib.disk import Disk, write_sysfs_file

# This class impement minimal binding of lio-targetcli
# Not for production usage, only for integration testing


class TargetCLI:
    _logger = logging.getLogger('diskmanager.lib.targetcli')

    @staticmethod
    def _do_exec(cmd):
        TargetCLI._logger.debug("exec : %s" % cmd)
        subprocess.check_call(cmd)

    @staticmethod
    def model_to_dev(model):
        for dname in Disk.list_disks():
            d = Disk(dname)
            if d.model == model:
                return d.device_path
        return ''

    def __init__(self, cmd='targetcli', clear=True):
        self._img_list = []
        self.cmd = cmd
        if clear:
            self.clear_config()

    def clear_config(self):
        cmd = [self.cmd, 'clearconfig', 'confirm=true']
        self._do_exec(cmd)

    def clear_images(self):
        for f in self._img_list:
            if os.path.exists(f):
                os.remove(f)

    def clear_all(self):
        self.clear_config()
        self.clear_images()

    def create_loopback(self, wwn=None):
        if wwn is None:
            wwn = generate_naa_wwn()
        cmd = [self.cmd, '/loopback/', 'create', wwn]
        TargetCLI._do_exec(cmd)
        return wwn

    def create_fileio_dev(self, name, size, image='', clear=True, rotational=True):
        lp = self.create_loopback()
        if not image:
            image = name

        if clear and os.path.exists(image):
            os.remove(image)
        cmd = [self.cmd, '/backstores/fileio', 'create']
        cmd.append('name=%s' % name)
        cmd.append('file_or_dev=%s' % image)
        cmd.append('size=%s' % size)
        TargetCLI._do_exec(cmd)
        # Set discard and write same support if necessary
        if not rotational:
            cmd = [self.cmd, '/backstores/fileio/%s' % name, 'set', 'attribute',
                   'emulate_tpu=1', 'emulate_tpws=1']
            TargetCLI._do_exec(cmd)

        cmd = [self.cmd]
        cmd.append('/loopback/%s/luns create /backstores/fileio/%s' % (lp, name))
        TargetCLI._do_exec(cmd)
        self._img_list.append(image)

        cmd = ['udevadm', 'settle']
        TargetCLI._do_exec(cmd)

        dev = TargetCLI.model_to_dev(name)
        # By default rotational is set. Clear it if requested
        if not rotational:
            d_path = os.path.join(Disk._SYS_BLOCK, os.path.basename(dev))
            write_sysfs_file('0', d_path, 'queue/rotational')
        return dev


def generate_naa_wwn():
    # see http://standards.ieee.org/develop/regauth/tut/fibre.pdf
    # 5 = IEEE registered
    # 001405 = OpenIB OUI (they let us use it I guess?)
    # rest = random
    return "naa.5001405" + uuid.uuid4().hex[-9:]
