from concurrent import futures
import argparse
import ConfigParser
import json
import os
import signal
import sys
import grpc

from infra.diskmanager.proto import diskman_pb2_grpc
from infra.diskmanager.lib.diskmanager import DiskManager
from infra.diskmanager.lib import consts
from infra.diskmanager.lib import logger


class DiskManagerServicer(diskman_pb2_grpc.DiskManagerServicer):
    def __init__(self, dm):
        self.dm = dm

    def ServerInfo(self, request, context):
        return self.dm.server_info(request, context)

    def ListDisks(self, request, context):
        return self.dm.list_disks(request, context)

    def FormatDisk(self, request, context):
        return self.dm.format_disk(request, context)

    def ListVolumes(self, request, context):
        return self.dm.list_volumes(request, context)

    def CreateVolume(self, request, context):
        return self.dm.create_volume(request, context)

    def DeleteVolume(self, request, context):
        return self.dm.delete_volume(request, context)

    def MountVolume(self, request, context):
        return self.dm.mount_volume(request, context)

    def UmountVolume(self, request, context):
        return self.dm.umount_volume(request, context)

    def SetIOLimitVolume(self, request, context):
        return self.dm.set_iolimit_volume(request, context)

    def GetYTMountedDevices(self, request, context):
        return self.dm.get_yt_devs(request, context)

    def GetHotSwapCreds(self, request, context):
        return self.dm.get_hot_swap_creds(request, context)

    def DaemonGetStat(self, request, context):
        return self.dm.daemon_get_stat(request, context)

    def DaemonUpdateCache(self, request, context):
        return self.dm.daemon_update_cache(request, context)

    def DaemonSetLoglevel(self, request, context):
        return self.dm.daemon_set_loglevel(request, context)


def get_config(args):
    int_names = ['stat_push_period', 'fstrim_period', 'quota_sync_period', 'log_size', 'log_rot', 'format_new_disk',
                 'format_new_disk_with_grub']
    conf = {}
    # Load from config, and then apply cmdline options
    if os.path.exists(args.config):
        try:
            if args.config.endswith('.json'):
                with open(args.config) as f:
                    load_conf = json.load(f)
                    conf.update(load_conf)
            else:
                cp = ConfigParser.ConfigParser()
                cp.read(args.config)
                if 'SERVER' in cp.sections():
                    conf.update(cp.items('SERVER'))
        except Exception, e:
            print("Fail to load config {}, err:{}".format(args.config, str(e)))
            sys.exit(1)
    # Covert to int if necessery
    for k, v in conf.iteritems():
        if k in int_names:
            conf[k] = int(v)

    if args.yasm_url:
        conf['yasm_url'] = args.yasm_url
    if args.stat_push_period:
        conf['stat_push_period'] = args.stat_push_period
    if args.fstrim_period:
        conf['fstrim_period'] = args.fstrim_period
    if args.quota_sync_period:
        conf['quota_sync_period'] = args.quota_sync_period
    if args.format_new_disk:
        conf['format_new_disk'] = args.format_new_disk
    if args.format_new_disk_with_grub:
        conf['format_new_disk_with_grub'] = args.format_new_disk_with_grub
    if args.default_io_scheduler:
        conf['default_io_scheduler'] = args.default_io_scheduler
    if args.logfile:
        conf['logfile'] = args.logfile
    if args.log_size:
        conf['log_size'] = args.log_size
    if args.log_rot:
        conf['log_rot'] = args.log_rot
    if args.ignore_dev:
        conf['ignore_dev'] = args.ignore_dev
    if args.include_dev:
        conf['include_dev'] = args.include_dev
    return conf


def sig_handler(signal, frame):
    print("Stop pid: %d by signal: %d" % (os.getpid(), signal))
    sys.exit(0)


def serve(args):
    config = get_config(args)
    hot_swap_socket = ""
    if args.service.startswith("unix:"):
        hot_swap_socket = args.service[5:]

    dm = DiskManager(config, hot_swap_socket=hot_swap_socket)
    signal.signal(signal.SIGTERM, sig_handler)
    server = grpc.server(futures.ThreadPoolExecutor(max_workers=1),
                         maximum_concurrent_rpcs=128)
    diskman_pb2_grpc.add_DiskManagerServicer_to_server(
        DiskManagerServicer(dm), server)
    server.add_insecure_port(args.service)
    server.start()
    # grpc doesn't allow to set unix socket permission, so set it manually
    # after server start
    if os.path.exists(consts.DEFAULT_SERVER_UNIX_SOCK):
        os.chown(consts.DEFAULT_SERVER_UNIX_SOCK, consts.DEFAULT_USER_ID, consts.DEFAULT_GROUP_ID)
        os.chmod(consts.DEFAULT_SERVER_UNIX_SOCK, consts.DEFAULT_SOCKET_PERMISSIONS)
    try:
        dm.serve()
    except KeyboardInterrupt:
        dm.stop_bg_threads(wait=True)
        server.stop(0)


def log_rot_t(string):
    try:
        v = int(string)
    except Exception:
        raise argparse.ArgumentTypeError('failed to parse "{}" as int'.format(string))
    if v <= 0:
        raise argparse.ArgumentTypeError('value must be >=0, got {} instead'.format(v))
    if v > 10:
        raise argparse.ArgumentTypeError('value must be <=10, got {} instead'.format(v))
    return v


def log_size_t(string):
    try:
        v = int(string)
    except Exception:
        raise argparse.ArgumentTypeError('failed to parse "{}" as int'.format(string))
    if v <= 0:
        raise argparse.ArgumentTypeError('value must be >=0, got {} instead'.format(v))
    if v > 200:
        raise argparse.ArgumentTypeError('value must be <=200, got {} instead'.format(v))
    return v


def init_parser(parser):
    parser.add_argument("-S", "--service", dest="service", type=str,
                        default=consts.DEFAULT_SERVER_ADDR)
    parser.add_argument("-c", "--config", dest="config", type=str,
                        default=consts.DEFAULT_SERVER_CONF)
    parser.add_argument("-Y", "--yasm-service", dest="yasm_url", type=str)
    parser.add_argument("--stat-period", dest="stat_push_period", type=int)
    parser.add_argument("--quota-sync-period", dest="quota_sync_period", type=int)
    parser.add_argument("--fstrim-period", dest="fstrim_period", type=int)
    parser.add_argument('--format-new-disk', dest='format_new_disk', default=False, action='store_true',  help='Format new empty disks')
    parser.add_argument('--format-new-disk-with-grub', dest='format_new_disk_with_grub', default=False, action='store_true',  help='Install grub on new disks')
    parser.add_argument('--default-io-scheduler', dest='default_io_scheduler', type=str,  help='Choose default io scheduler: [cfq|mq-deadline|bfq]')
    parser.add_argument("--ignore-dev", dest="ignore_dev", action="append",
                        type=str, default=[],
                        help='Override default ignore devices list')
    parser.add_argument("--include-dev", dest="include_dev", action="append",
                        type=str, default=[], help="remove devices from ignore list")
    parser.add_argument("-L", "--logfile", dest="logfile", type=str,
                        default=consts.DEFAULT_SERVER_LOG)
    parser.add_argument("--log_size_mb", dest="log_size", type=log_size_t,
                        default=10)
    parser.add_argument("--log_rotation", dest="log_rot", type=log_rot_t,
                        default=8)
    parser.add_argument("-v", "--verbose", dest="verbose",
                        action="count", default=2,
                        help="increases log verbosity for each occurence.")


if __name__ == '__main__':
    try:
        parser = argparse.ArgumentParser(description='Diskmanager service',
                                         epilog='Documentation: %s' % consts.WIKI_URL)
        init_parser(parser)
        args = parser.parse_args(sys.argv[1:])
        l = logger.setup_logger('diskmanager', args.verbose)
        logger.log_add_file(l, args.logfile, args.log_size * 1024 ** 2, args.log_rot)
        serve(args)
    except KeyboardInterrupt:
        raise SystemExit(0)
