#!/bin/bash

exec &> >(tee -a run.log)

log(){
    echo -e "[$(date +%T)] $1"
}
fail(){
    echo -e "\033[0;31m[$(date +%T)] $1\033[0m"
    exit 1
}

[[ $1 == "--help" ]] && echo -e "Usage: \033[1m$0 /dev/sda\033[0m (for sata ssd) or \033[1m$0 /dev/nvme0n1p2\033[0m (for nvme ssd)" && exit 0
# minimal checks
[[ -f fio.static ]] || fail "There is no fio binary. Run prepare_env.sh to get lastest fio version."
[[ -z $1 ]] && fail "There is no device argument. Example: $0 /dev/sda OR $0 /dev/nvme0n1p2" 
[[ -b $1 ]] || fail "There is no device $1 or it's not block device"


DEVICE_PATH=$1
DEVICE_NAME=$(echo $DEVICE_PATH | awk -F'/' '{print $NF}' | head -c 7)
DATE=$(date +%F_%H-%m)
HOST_SERIAL=$(sudo dmidecode --type system | grep Serial | awk '{print $NF}')
RESULTS_DIRNAME="${HOST_SERIAL}_${DEVICE_NAME}_${DATE}"

# preparing result directories
if [[ $DEVICE_NAME == *'nvme'* ]]; 
then
    DEVICE_TYPE='nvme'
else
    [[ `cat /sys/block/${DEVICE_NAME}/queue/rotational` == "1" ]] && fail "$DEVICE_PATH should be ssd"
    DEVICE_TYPE='ssd'
fi

mkdir -p results/$RESULTS_DIRNAME/info
mkdir -p results/$RESULTS_DIRNAME/jsons
log "Created result directory \033[1mresults/$RESULTS_DIRNAME\033[0m"

# saving all stuff about host and device
sudo  dmidecode --type system > results/$RESULTS_DIRNAME/info/host.dmidecode
uname -a > results/$RESULTS_DIRNAME/info/host.uname
[[ $DEVICE_TYPE == 'nvme' ]] && sudo nvme list $DEVICE_PATH -o json > results/${RESULTS_DIRNAME}/info/${DEVICE_NAME}.nvme_list
[[ $DEVICE_TYPE == 'ssd' ]] && sudo hdparm -I $DEVICE_PATH > results/${RESULTS_DIRNAME}/info/${DEVICE_NAME}.hdparm
log "Collected host and device info"

# running fio
if [ $DEVICE_TYPE == 'ssd' ];
then
    COMMAND="FIO_TEST_DEVICE=$DEVICE_PATH sudo -E ./fio.static fio_configs/scenario_sata.fio --output-format=json -o results/$RESULTS_DIRNAME/jsons/fio_out_sata.json"
    log "Running fio scenario (ETA ~ 3h): $COMMAND"
else
    COMMAND="FIO_TEST_DEVICE=$DEVICE_PATH NUMJOBS=2 sudo -E ./fio.static fio_configs/scenario_nvme.fio --output-format=json -o results/$RESULTS_DIRNAME/jsons/fio_out_nvme_num2.json; FIO_TEST_DEVICE=$DEVICE_PATH SIZE=$SIZE NUMJOBS=4 sudo -E ./fio.static fio_configs/scenario_nvme.fio --output-format=json -o results/$RESULTS_DIRNAME/jsons/fio_out_nvme_num4.json;"
    log "Running fio scenario (ETA ~ 6h): $COMMAND"
fi

eval $COMMAND

# packing results
tar -czf ${RESULTS_DIRNAME}.tar.gz fio_configs/ results/$RESULTS_DIRNAME/
log "Testing is done. Results archive \033[1m${RESULTS_DIRNAME}.tar.gz\033[0m"
