#!/bin/bash

usage()
{
    echo "Usage: find_package.sh [options] <package name>=<version>"
    echo "       find_package.sh [options] <package name>_<version>"
    echo "       find_package.sh [options] <-n package name> [-v version]"
    echo
    echo "Options:"
    echo "       -d             - debug (show search url)"
    echo "       -e <branch>    - search in selected branch"
    echo "       -f             - strict search anyway"
    echo "       -h             - this help"
    echo "       -i             - inaccurate search anyway"
    echo "       -j             - mongrel output (show json)"
    echo "       -l             - also search on FS (via old find_packages.sh)"
    echo "       -m             - show exactly selected name and/or version"
    echo "       -n <name>      - package name"
    echo "       -r <repo>      - search in selected repo"
    echo "       -s             - show sources"
    echo "       -u             - show urls for downloading"
    echo "       -v <version>   - package version"
    echo
    echo "       --fs-only      - search only on filesystem"
    echo
    echo "Examples:"
    echo "       find_package.sh nginx"
    echo "       find_package.sh nginx=1.8.1-1.yandex.15"
    echo "       find_package.sh -r yandex-trusty nginx_1.8.1-1.yandex.15"
    echo "       find_package.sh -v 1.8.1-1.yandex.15"
    echo "       find_package.sh -u -s nginx=1.8.1-1.yandex.15"
    echo "       find_package.sh -n nginx -v 1.8.1-1.yandex.15 -s"
    echo "       find_package.sh nginx=1.8.1-1.yandex -s -i"
    echo
    exit 1
}

while [ $# -gt 0 ]; do
  case $1 in
    -n)
      _name="pkg=$2"

      if [ "${_name}" == "pkg=" -o "${_name:4:1}" == "-" ]; then
          echo "Error: Parameter -n with empty value."
          echo
          usage
      fi

      _name_from_param="true"
      shift; shift
      ;;
    -v)
      _version="ver=$2"

      if [ "${_version}" == "ver=" -o "${_version:4:1}" == "-" ]; then
          echo "Error: Parameter -v with empty value."
          echo
          usage
      fi

      _version_from_param="true"
      shift; shift
      ;;
    -r)
      _repo="repo=$2"

      if [ "${_repo}" == "repo=" -o "${_repo:5:1}" == "-" ]; then
          echo "Error: Parameter -r with empty value."
          echo
          usage
      fi

      shift; shift
      ;;
    -e)
      _env="env=$2"

      if [ "${_env}" == "env=" -o "${_env:4:1}" == "-" ]; then
          echo "Error: Parameter -e with empty value."
          echo
          usage
      fi

      shift; shift
      ;;
    -u)
      _withurls="&withurls=true"
      shift
    ;;
    -s)
      _sources="&sources=true"
      shift
    ;;
    -i)
      _strict="&strict=false"
      _force_strict="false"
      shift
    ;;
    -l)
      _search_on_fs="true"
      shift
    ;;
    -f)
      _force_strict="true"
      shift
    ;;
    --fs-only)
      _fs_only="true"
      shift
    ;;
    -d)
      _debug="true"
      shift
    ;;
    -j)
      _mongrel_output="true"
      shift
    ;;
    -m)
      _match_search="true"
      shift
    ;;
    -h)
      usage
      shift
    ;;
    *)
      _name="$1"

      if [ "${_name:0:1}" == "-" ]; then
          echo "Error: Unknown parameter ${_name}."
          echo
          usage
      fi

      _old_param="true"
      shift
    ;;
  esac
done

# Abort if all next parameters are empty. Also we skip -e option at
# this time. Fix me.

if [ -z "${_name}" -a -z "${_version}" -a -z "${_repo}" ]; then
    echo "Error: Name, version or repo name not defined"
    echo
    usage
fi

# Process parameter like <name>=<version> as it was in previous version
# of this script

if [ "x${_old_param}" == "xtrue" ]; then
    if [ "x${_version}" == "x" -a "x${_name_from_param}" != "xtrue" ]; then
        if [ $(echo ${_name} | egrep "=") ]; then
            _version="ver=$(echo ${_name} | awk -F= '{ print $NF }')"
            _name="pkg=$(echo ${_name} | awk -F= '{ print $1 }')"
        elif [ $(echo ${_name} | egrep "_") ]; then
            _version="ver=$(echo ${_name} | awk -F_ '{ print $NF }')"
            _name="pkg=$(echo ${_name} | awk -F_ '{ print $1 }')"
        else
            _name="pkg=${_name}"
        fi
    else
        echo "Error: Do not use -v or -n options with <package>=<version>"
        echo
        usage
    fi
fi

# Some logics in strict parameter

if [[ "${_name}" == *+* || "${_version}" == *+* ]]; then
	_strict="&strict=true"
elif [ -z "${_name}" -o -z "${_version}" ]; then
    _strict="&strict=false"

elif [ "x${_force_strict}" != "xtrue" ]; then
    if [ -n "${_name}" -a -n "${_version}" ]; then
        _strict="&strict=true"
    else
        _strict="&strict=false"
    fi
else
    _strict="&strict=true"
fi

if [ "x${_force_strict}" == "xfalse" ]; then
        _strict="&strict=false"
elif [ "x${_force_strict}" == "xtrue" ]; then
    _strict="&strict=true"
fi

# Human output or json

if [ "x${_mongrel_output}" == "xtrue" ]; then
    _human="&human=false"
else
    _human="&human=true"
fi

# Drop excess ampersands and replace + with %2B

_string=$(echo "${_name}&${_version}&${_repo}&${_env}&${_strict}&${_human}&${_sources}&${_withurls}" | \
    sed 's@^&@@;s@&$@@;s@&$@@;s@&&*@\&@g;s@+@%2B@g')

# Drop epoch for pattern and filesystem search

_version=$(echo ${_version} | awk -F: '{ print $NF }')

if [ "x${_fs_only}" != "xtrue" ]; then
    if [ "${_mongrel_output}" != "true" ]; then
        echo
        echo "Searching in Cacus/MDS:"
        echo
    fi

    if [[ "x${_match_search}" != "xtrue" ]] || [[ "x${_version}" == "x" && "x${_name}" == "x" ]]; then
        response=$(curl -f -s  --write-out "%{http_code}" "http://dist.yandex.ru/api/v1/search?${_string}")
        result=${response%???}
        status=${response: -3}

    else
        if [ "x${_version}" != "x" -a "x${_name}" != "x" ]; then
            PATTERN="${_name#pkg=}_${_version#ver=}(_|.dsc|.tar|.orig|.diff|.debian.git)"
        elif [ "x${_version}" == "x" -a "x${_name}" != "x" ]; then
            PATTERN=" ${_name#pkg=}_"
        elif [ "x${_version}" != "x" -a "x${_name}" == "x" ]; then
            PATTERN="_${_version#ver=}(_|.dsc|.tar|.orig|.diff|.debian|.git)"
        fi
        PATTERN=$(echo $PATTERN | sed 's@\.@\\.@g')

        response=$(curl -f --silent --write-out "%{http_code}" "http://dist.yandex.ru/api/v1/search?${_string}")
        body=${response%???}
        status=${response: -3}
        result=$(egrep --color=never " repo |----|$PATTERN" <<< "$body")

    fi

    if [ $status == '404' ]; then
        echo "No packages found"
    elif [ $status == '200' ]; then
        echo -e "$result"
    else
        echo "API request error"
    fi
fi

if [ "x${_search_on_fs}" == "xtrue" -o "x${_fs_only}" == "xtrue" ]; then
    echo
    echo "Searching on filesystem via old find_packages.sh:"
    echo

    if [ "x${_force_strict}" == "xfalse" ]; then
        old_find_package.sh ${_name#pkg=}_${_version#ver=}
    else
        old_find_package.sh ${_name#pkg=}=${_version#ver=}
    fi
fi

if [ "x${_debug}" == "xtrue" ]; then
    echo
    echo "Search URL:"
    echo
    echo "curl -f -s \"http://dist.yandex.ru/api/v1/search?${_string}\""
    echo
fi

