from __future__ import absolute_import

import argparse
import sys
import yaml
import subprocess
import logging
import logging.handlers

from infra.dist.cacus.lib import constants


def strtobool(val):
    """
    Convert a string representation of truth to true (1) or false (0).

    True values are 'y', 'yes', 't', 'true', 'on', and '1'; false values
    are 'n', 'no', 'f', 'false', 'off', and '0'.  Raises ValueError if
    'val' is anything else.
    """
    val = val.lower()
    if val in ('y', 'yes', 't', 'true', 'on', '1'):
        return 1
    elif val in ('n', 'no', 'f', 'false', 'off', '0'):
        return 0
    else:
        raise ValueError("invalid truth value %r" % (val,))


def load_config(config_file):
    with open(config_file) as cfg:
        config = yaml.safe_load(cfg)
    process = subprocess.Popen(
        [
            'gpg',
            '--list-keys',
            config['gpg']['signer']
        ],
        stdout=subprocess.PIPE,
        stderr=subprocess.PIPE
    )
    stdout, stderr = process.communicate()
    if stderr or not stdout:
        msg = "Cannot find suitable keys for %s", config['gpg']['signer']
        logging.critical(msg)
        # sys.exit(1)
    return config


def setup_logging(conf, action_name):
    root_logger = logging.getLogger()
    root_logger.setLevel(getattr(logging, conf['level']))
    log_formatter = logging.Formatter(
        "%(asctime)s [%(levelname)-4.4s](%(process)d) %(name)s: %(message)s")
    dst = conf['destinations']
    root_logger.handlers = []
    if dst['console']:
        h = logging.StreamHandler()
        h.setFormatter(log_formatter)
        root_logger.addHandler(h)
    if dst['file']:
        h = logging.handlers.WatchedFileHandler(dst['file_pattern'].format(action_name=action_name))
        h.setFormatter(log_formatter)
        root_logger.addHandler(h)
    if dst['syslog']:
        h = logging.handlers.SysLogHandler(facility=dst['syslog'])
        h.setFormatter(
            logging.Formatter("[%(levelname)-4.4s] %(name)s: %(message)s"))
        root_logger.addHandler(h)


def main():
    def under_construction(*args, **kwargs):
        print 'This option is broken for current moment'
        sys.exit(1)

    parser = argparse.ArgumentParser()
    parser.add_argument('-v', '--log-level', type=str, default=None,
                        help='Log level(from python logging package) default: read from config')

    parser.add_argument('-c', '--config', type=str, default='/etc/cacus.yaml',
                        help='Config file (default /etc/cacus.yaml')

    subparsers = parser.add_subparsers(help='sub-command --help')
    parser_upload = subparsers.add_parser('upload', help='Upload package')
    parser_upload.set_defaults(action="under_construction")
    parser_repo_summary = subparsers.add_parser('repo-summary', help='Repository summary')
    parser_repo_summary.set_defaults(action='repo-summary')
    parser_repo_summary.add_argument('repo', help='repository name')

    cm_main_hlp = 'clean-branch - delete branches with names like [a-z]*2remove'
    parser_clean_branch = subparsers.add_parser('clean-branch', help=cm_main_hlp)
    parser_clean_branch.set_defaults(action="clean-branch")
    parser_clean_branch.add_argument('repo', type=str, help='Name of repository')
    parser_clean_branch.add_argument('branch', type=str, help='Name of branch')

    dm_main_hlp = 'Dmove package - move package between environments'
    parser_dmove = subparsers.add_parser('dmove', help=dm_main_hlp)
    parser_dmove.set_defaults(action="dmove")
    parser_dmove.add_argument('repo', type=str,
                              help='Name of repository, containing package')
    dm_dst_hlp = 'Name of environment to move to (unstable, stable, etc...)'
    parser_dmove.add_argument('destination', type=str, help=dm_dst_hlp)
    parser_dmove.add_argument('pkg', type=str,
                              help='Move package with this name')
    parser_dmove.add_argument('version', type=str,
                              help='Move this version of package')
    dm_src_hlp = 'Name of environment to move from (unstable, stable, etc...)'
    parser_dmove.add_argument('source', type=str, help=dm_src_hlp)
    parser_dmove.add_argument('--skipUpdateMeta', default='false',
                              type=strtobool,
                              help='For internal usage')

    bm_main_hlp = 'Bmove package - move package between repositories'
    parser_bmove = subparsers.add_parser('bmove', help=bm_main_hlp)
    parser_bmove.set_defaults(action="bmove")
    bm_env_hlp = ('Name of environment, containing package '
                  '(unstable, stable, etc...)')
    parser_bmove.add_argument('env', type=str, help=bm_env_hlp)
    parser_bmove.add_argument('destination', type=str,
                              help='Name of repository to move to')
    parser_bmove.add_argument('pkg', type=str,
                              help='Move package with this name')
    parser_bmove.add_argument('version', type=str,
                              help='Move this version of package')
    parser_bmove.add_argument('source', type=str,
                              help='Name of repository to move from')

    bc_main_hlp = 'Bcopy package - copy package from one repository to another'
    parser_bcopy = subparsers.add_parser('bcopy', help=bc_main_hlp)
    parser_bcopy.set_defaults(action="bcopy")
    bc_env_hlp = ('Name of environment, containing package '
                  '(unstable, stable, etc...)')
    parser_bcopy.add_argument('env', type=str,
                              help=bc_env_hlp)
    parser_bcopy.add_argument('destination', type=str,
                              help='Name of repository to copy to')
    parser_bcopy.add_argument('pkg', type=str,
                              help='Copy package with this name')
    parser_bcopy.add_argument('version', type=str,
                              help='Copy this version of package')
    parser_bcopy.add_argument('source', type=str,
                              help='Name of repository to copy from')

    parser_rmove = subparsers.add_parser('rmove', help='Remove package')
    parser_rmove.set_defaults(action="rmove")
    parser_rmove.add_argument('repo', type=str,
                              help='Remove package from this repository')
    rm_env_hlp = ('Remove package from this environment '
                  '(unstable, stable, etc...)')
    parser_rmove.add_argument('env', type=str,
                              help=rm_env_hlp)
    parser_rmove.add_argument('pkg', type=str,
                              help='Remove package with this name')
    parser_rmove.add_argument('version', type=str,
                              help='Remove this version of package')

    parser_add_repo = subparsers.add_parser('add-repo', help='Add repository')
    parser_add_repo.set_defaults(action="add_repo")
    parser_add_repo.add_argument('repo', type=str,
                                 help='Name of new repository')
    parser_add_repo.add_argument('descr', type=str,
                                 help='Description for new repository')

    dd_main_hlp = 'Start incoming watcher daemon'
    parser_duploader_daemon = subparsers.add_parser('duploader-daemon',
                                                    help=dd_main_hlp)
    parser_duploader_daemon.set_defaults(action="duploader-daemon")
    rd_main_hlp = 'Start daemon, that serves user requests'
    parser_repo_daemon = subparsers.add_parser('repo-daemon', help=rd_main_hlp)
    parser_repo_daemon.set_defaults(action="repo-daemon")
    rd_main_hlp = 'Start daemon, that serves Staff GPG keys cache'
    parser_gpg_daemon = subparsers.add_parser('gpg-daemon', help=rd_main_hlp)
    parser_gpg_daemon.set_defaults(action="gpg-daemon")
    parser_gpg_daemon.add_argument('--force', '-f', action='store_true', default=False,
                                   help='Force update of all keyrings')
    ur_main_hlp = 'Update repository metadata'
    parser_update_repo = subparsers.add_parser('update-repo', help=ur_main_hlp)
    parser_update_repo.set_defaults(action="update-repo")
    parser_update_repo.add_argument('--force', '-f', action="store_true", default=False)
    parser_update_repo.add_argument('repo', type=str,
                                    help='Update repo metadata with this name')
    ur_env_hlp = ('Update metadata for this environment '
                  '(unstable, stable, etc...)')
    parser_update_repo.add_argument('env', type=str, help=ur_env_hlp)
    ur_arch_hlp = 'Update metadata for this architecture'
    parser_update_repo.add_argument('arch', type=str,
                                    help=ur_arch_hlp)
    parser_sources_daemon = subparsers.add_parser(
        'sources-daemon',
        help='Updates sources files in cacus. Not to be run by user.'
    )
    parser_sources_daemon.set_defaults(action="sources-daemon")
    parser_packages_daemon = subparsers.add_parser(
        'packages-daemon',
        help='Updates packages files in cacus. Not to be run by user.'
    )
    parser_packages_daemon.set_defaults(action="packages-daemon")
    parser_update_upstream_list = subparsers.add_parser(
        'update-upstream-list',
        help='Updates upstream packages list. Not to be run by user.'
    )
    parser_update_upstream_list.set_defaults(action="update-upstream-list")
    parser_exec_defered_rmove = subparsers.add_parser(
        'exec-defered-rmove',
        help='Runs defered key delete job. Not to be run by user.'
    )
    parser_exec_defered_rmove.set_defaults(action="exec-defered-rmove")

    ir_main_hlp = 'Import mounted dist.yandex.ru repo'
    parser_import_repo = subparsers.add_parser('import-repo', help=ir_main_hlp)
    parser_import_repo.set_defaults(action="import-repo")
    parser_import_repo.add_argument('repo_url', type=str,
                                    help='Path to mounted repository')
    parser_import_repo.add_argument('repo', type=str,
                                    help='Name of repository to import to')
    ir_env_hlp = 'Name of environment to import to (unstable, stable, etc...)'
    parser_import_repo.add_argument('env', type=str, help=ir_env_hlp)

    parser_drop = subparsers.add_parser('drop-repo', help='Drop repository')
    parser_drop.set_defaults(action="drop-repo")
    parser_drop.add_argument('repo', type=str,
                             help='Name of repository to be dropped')

    parser_mask = subparsers.add_parser('mask-repo', help='Mask repository')
    parser_mask.set_defaults(action="mask-repo")
    parser_mask.add_argument('repo', type=str,
                             help='Name of repository to be masked')

    parser_list = subparsers.add_parser('list-repos', help='List repositories')
    parser_list.set_defaults(action="list-repos")

    parser_stats_daemon = subparsers.add_parser('stats-daemon', help='Start stats daemon')
    parser_stats_daemon.set_defaults(action='stats-daemon')

    parser_zk_stats_daemon = subparsers.add_parser('zk_stats-daemon', help='Zookeeper stats daemon(localhost)')
    parser_zk_stats_daemon.set_defaults(action='zk_stats-daemon')

    parser_mongo_stats_daemon = subparsers.add_parser('mongo_stats-daemon', help='MongoDB stats daemon(localhost)')
    parser_mongo_stats_daemon.set_defaults(action='mongo_stats-daemon')

    parser_repo_cleaner = subparsers.add_parser('repo-cleaner', help='Repository Cleaner')
    parser_repo_cleaner.set_defaults(action='repo-cleaner')
    parser_repo_cleaner_subparser = parser_repo_cleaner.add_subparsers(help='sub-actions')
    parser_repo_cleaner_prepare = parser_repo_cleaner_subparser.add_parser('prepare', help='prepare cleanup batch')
    parser_repo_cleaner_prepare.set_defaults(sub_action='prepare')
    parser_repo_cleaner_prepare.add_argument('--repos-config', help='Repository cleaner configuration(yaml)',
                                             default='cleanup.yaml')
    parser_repo_cleaner_prepare.add_argument('--repo', '-r', help='Repo(s) to work with', required=False, nargs='*',
                                             default=['all'])
    parser_repo_cleaner_prepare.add_argument('--output-dir', '-o', help='Batche(es) output dir', required=False,
                                             default='out')
    parser_repo_cleaner_apply = parser_repo_cleaner_subparser.add_parser('apply', help='apply cleanup batch')
    parser_repo_cleaner_apply.set_defaults(sub_action='apply')
    parser_repo_cleaner_apply.add_argument('--batch', '-b', help='Batch(es) to apply', nargs='*', required=True)
    parser_repo_cleaner_apply.add_argument('--yes-i-really-want-lsr-ticket', help='really apply batch on database',
                                           action='store_true', default=False)
    parser_repo_cleaner_apply.add_argument('--keep-going', '-k', help='keep going if dmove fails',
                                           action='store_true', default=False)
    parser_repo_cleaner_recycle = parser_repo_cleaner_subparser.add_parser('recycle', help='recycle packages')
    parser_repo_cleaner_recycle.set_defaults(sub_action='recycle')
    parser_repo_cleaner_recycle.add_argument('--repos-config', help='Repository cleaner configuration(yaml)',
                                             default='cleanup.yaml')
    parser_repo_cleaner_recycle.add_argument('--repo', '-r', help='Repo(s) to work with', required=False, nargs='*',
                                             default=['all'])
    args = parser.parse_args()

    # dirty hacks for initialization
    action_name = args.action
    config = load_config(args.config)
    if args.log_level is not None:
        config['logging']['level'] = args.log_level
    setup_logging(config['logging'], action_name)
    from infra.dist.cacus.lib.dbal import mongo_connection
    mongo_connection.configure(config['metadb'])
    # compatibility crutch
    import infra.dist.cacus.lib.common as common
    common.config = config
    common.action_name = action_name

    import infra.dist.cacus.lib.loader
    infra.dist.cacus.lib.loader.load_plugins()
    from infra.dist.cacus.lib import repo_daemon, restrict_daemon, dist_importer, repo_manage
    import infra.dist.cacus.lib.daemon.duploader.duploader as duploader
    func_args = {}
    for key, value in vars(args).iteritems():
        if not (key in ("log", "config", "verbosity", "action", 'log_level')):
            func_args[key] = value

    # launch daemons actions

    daemonize, daemon_user, daemon_group = common.get_daemon_params(common.config)
    if args.action == 'stats-daemon':
        import infra.dist.cacus.lib.daemon.stats.daemon
        if daemonize:
            common.daemonize(daemon_user, daemon_group, infra.dist.cacus.lib.daemon.stats.daemon.main)
        else:
            infra.dist.cacus.lib.daemon.stats.daemon.main()
    if args.action == "duploader-daemon":
        if daemonize:
            common.daemonize(daemon_user, daemon_group, duploader.start_duploader)
        else:
            duploader.start_duploader()
    elif args.action == "repo-daemon":
        if daemonize:
            common.daemonize(daemon_user, daemon_group, repo_daemon.start_daemon)
        else:
            repo_daemon.start_daemon()
    elif args.action == "gpg-daemon":
        from infra.dist.cacus.lib.daemon.gpg.gpg_daemon import start_gpg_daemon
        if daemonize:
            common.daemonize(daemon_user, daemon_group, start_gpg_daemon, func_args=(True,))
        else:
            start_gpg_daemon()
    elif args.action == "sources-daemon":
        if daemonize:
            common.daemonize(daemon_user, daemon_group, repo_manage.update_sources_files)
        else:
            repo_manage.update_sources_files()
    elif args.action == "packages-daemon":
        if daemonize:
            common.daemonize(daemon_user, daemon_group, repo_manage.update_all_packages_metadata)
        else:
            repo_manage.update_all_packages_metadata()
    if args.action == 'zk_stats-daemon':
        import infra.dist.cacus.lib.daemon.zk_stats as zk_stats
        if daemonize:
            common.daemonize(daemon_user, daemon_group, zk_stats.main)
        else:
            zk_stats.main()
    if args.action == 'mongo_stats-daemon':
        import infra.dist.cacus.lib.daemon.mongo_stats as mongo_stats
        if daemonize:
            common.daemonize(daemon_user, daemon_group, mongo_stats.main)
        else:
            mongo_stats.main()
        exit(0)

    # in-place actions
    if args.action == "under_construction":
        under_construction(**func_args)
    elif args.action == "repo-info":
        # TODO: FIXME
        exit(-1)
    elif args.action == 'clean-branch':
        res = repo_manage.clean_branch(**func_args)
        if res['result'] == constants.Status.NOT_FOUND:
            sys.exit(1)
        if res['result'] == constants.Status.TIMEOUT:
            sys.exit(77)
    elif args.action == "dmove":
        func_args['skipUpdateMeta'] = bool(func_args['skipUpdateMeta'])
        res = repo_manage.dmove_package(**func_args)
        if res['result'] == constants.Status.NOT_FOUND:
            sys.exit(1)
        if res['result'] == constants.Status.TIMEOUT:
            sys.exit(77)
    elif args.action == "bmove":
        res = repo_manage.bmove_package(**func_args)
        if res['result'] == constants.Status.NOT_FOUND:
            sys.exit(1)
        if res['result'] == constants.Status.TIMEOUT:
            sys.exit(77)
    elif args.action == "bcopy":
        func_args['delete'] = False
        res = repo_manage.bmove_package(**func_args)
        if res['result'] == constants.Status.NOT_FOUND:
            sys.exit(1)
        if res['result'] == constants.Status.TIMEOUT:
            sys.exit(77)
    elif args.action == "rmove":
        res = repo_manage.rmove_package(**func_args)
        if res['result'] == constants.Status.NOT_FOUND:
            sys.exit(1)
        if res['result'] == constants.Status.TIMEOUT:
            sys.exit(77)
    elif args.action == "add_repo":
        repo_manage.add_repository(**func_args)
    elif args.action == "update-repo":
        log = logging.getLogger(__name__)
        repo = func_args['repo']
        env = func_args['env']
        arch = func_args['arch']
        try:
            with common.RepoLock(repo, env, nonblocking=False):
                repo_manage.update_repo_metadata(**func_args)
                msg = ('Packages files for {}/{}/{} has '
                       'been successfully updated'.format(repo, env, arch))
                log.info(msg)
        except common.MetadataIsBusy:
            msg = ('Packages files for {}/{}/{} seems to '
                   'be already updating just right now. '
                   'Skipping'.format(repo, env, arch))
            log.info(msg)
            sys.exit(77)
    elif args.action == "update-upstream-list":
        restrict_daemon.update_upstream_packages_list()
    elif args.action == "exec-defered-rmove":
        repo_manage.exec_defered_rmove()
    elif args.action == "import-repo":
        dist_importer.import_repo(**func_args)
    elif args.action == "drop-repo":
        repo_manage.rmove_repository(**func_args)
    elif args.action == "mask-repo":
        repo_manage.mask_repository(**func_args)
    elif args.action == "list-repos":
        repo_manage.list_repos(**func_args)
    elif args.action == "repo-cleaner":
        import infra.dist.cacus.lib.tools.repo_cleaner.main
        infra.dist.cacus.lib.tools.repo_cleaner.main.main(args)
