from __future__ import absolute_import

import datetime
import hashlib
import os
import pwd
import re
import subprocess
import sys
import time
import traceback
import logging
from functools import partial
from random import randint

import enum

from infra.dist.cacus.lib import constants
import infra.dist.cacus.lib.common
import infra.dist.cacus.lib.daemon.duploader.legacy as legacy
from infra.dist.cacus.lib import repo_manage
from infra.dist.cacus.lib.daemon.duploader.exceptions import DebWaitAttemptsExceeded
from infra.dist.cacus.lib.daemon.duploader.stats import duploader_package_upload_timing_hgram, \
    duploader_rejected_package_count, \
    duploader_total_package_count, \
    duploader_uploaded_package_count
from infra.dist.cacus.lib.daemon.duploader.worker import Worker
from infra.dist.cacus.lib.utils.microdinstall import change_file
from infra.dist.cacus.lib.notifications import factory as notifications
from infra.dist.cacus.lib.stats.timing_decorator import callable_timing_hram
from infra.dist.cacus.lib.dbal import ubuntu_upstream

log = logging.getLogger(__name__)


class PseudoEvent:
    def __init__(self, path, pathname):
        self.path = path
        self.pathname = pathname


class UploadStatus(enum.Enum):
    UNKNOWN = 0
    CLEANUP_NEEDED = 1
    REJECT_NEEDED = 2
    MOVE_TO_DINSTALL_NEEDED = 3


class ChangesWorker(Worker):
    def __init__(self, changes_file_path, upstream_dists_getter, repos_conf=None):
        self.changes_file_path = changes_file_path
        self.parsed_chages = None
        self.notification = None
        self.incoming_files = None
        self.upload_status = UploadStatus.UNKNOWN
        self.upstream_dists_getter = upstream_dists_getter
        self.repos_conf = repos_conf if repos_conf is not None else {}
        self.repo = self.extract_repo()
        self.repo_conf = self.repos_conf[self.repo]
        self.yndx_dinst_conf = legacy.read_yandex_dinstall_config(self.repo)
        self.mode = 'compatibility' if self.yndx_dinst_conf else 'cacus_only'
        self.uploaded_files = set()

    def cleanup(self):
        if self.upload_status == UploadStatus.CLEANUP_NEEDED:
            for f in self.incoming_files:
                try:
                    os.unlink(f)
                except OSError as error:
                    log.critical('%s: cannot unlink file: %s error: %s', self.changes_file_path, f, error)
        elif self.upload_status == UploadStatus.REJECT_NEEDED:
            for f in self.incoming_files:
                self._move_to_reject(f)
        elif self.upload_status == UploadStatus.MOVE_TO_DINSTALL_NEEDED:
            for f in self.incoming_files:
                self._move_to_yandex_dinstall(f)
        else:
            log.critical("UNKNOWN STATUS FOR: %s", self.changes_file_path)
        if self.notification is not None:
            self.notification.send()

    def run(self):
        try:
            path = os.path.dirname(self.changes_file_path)
            event = PseudoEvent(path, self.changes_file_path)
            self._processChangesFile(event)
        except DebWaitAttemptsExceeded as tm:
            log.error(tm)
            self.upload_status = UploadStatus.CLEANUP_NEEDED

    def extract_repo(self):
        parts = self.changes_file_path.split('/')
        return parts[-4]

    def _get_md5_sum(self, filename):
        sum = hashlib.md5()
        log.debug("Generate %s (python-internal) for %s" % (type, filename))
        with open(filename, 'r') as f:
            buf = f.read(8192)
            while buf != '':
                sum.update(buf)
                buf = f.read(8192)
        return sum.hexdigest()

    def _gpg_check(self, filename, repo):
        if repo not in legacy.keyring_cache:
            repo = 'main_keyring'
        keyring_file = legacy.keyring_cache[repo]['keyring']
        legacy.keyring_cache_rw_locks[repo].acquire_read()
        command = 'gpgv --keyring {} {}'.format(keyring_file, filename)
        log.debug('checking changes with gpg: {}'.format(command))
        proc = subprocess.Popen(
            command,
            shell=True,
            env=dict(os.environ, LANG="C"),
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE
        )
        proc.wait()
        output = proc.communicate()[1]  # because gpgv type messages on stderr
        legacy.keyring_cache_rw_locks[repo].release_read()
        if proc.returncode:
            msg = 'Got non-zero return code. Output of gpgv was: \n{}'.format(
                output
            )
            raise RuntimeError(msg)
        match = re.search('gpgv: Good signature from "(?P<signer>.*)"', output)
        signer = match.group('signer')
        return signer

    def _check_for_upstream_debs(self, deb_files, upstream_store=ubuntu_upstream.default_store):
        deb_names = []
        splitted_deb_names = map(
            lambda x: os.path.basename(x).rpartition('.'), deb_files
        )
        for fqpn, delim, extension in splitted_deb_names:
            name = fqpn.partition('_')[0]
            dotted_ext = '{}{}'.format(delim, extension)
            if dotted_ext in constants.deb_extensions:
                deb_names.append(name)
            else:
                msg = '{} seems to be some sort of source file.' \
                      ' It will not participate in upstream check.' \
                      ''.format(''.join((fqpn, delim, extension)))
                log.debug(msg)

        existing = upstream_store.find_one(dist=self.repo.replace('yandex-', ''), package={'$in': deb_names})

        return existing.package if existing else None

    def _move_to_yandex_dinstall(self, path):
        filename = os.path.basename(path)
        cacus_dir = os.path.dirname(path)
        yandex_dinstall_path = os.path.abspath('{}../../yandex-dinstall/incoming/{}'.format(
            cacus_dir, filename))
        try:
            os.rename(path, yandex_dinstall_path)
        except OSError as error:
            log.critical('error moving %s to yandex-dinstall location: %s', path, error)

    def _move_to_workdir(self, files):
        result = []
        work_path = os.path.abspath(os.path.join(os.path.dirname(self.changes_file_path), '../work'))
        log.debug('moving files to: %s', work_path)
        for f in files:
            filename = os.path.basename(f)
            log.debug('moving %s to %s', f, work_path)
            try:
                dst = os.path.join(work_path, filename)
                os.rename(f, dst)
                result.append(dst)
            except OSError:
                log.exception('cannot move %s to %s', f, work_path)
                result = []
                break
        return result

    def _move_to_reject(self, path):
        reject_path = os.path.abspath(os.path.join(os.path.dirname(path), '../REJECT', os.path.basename(path)))
        try:
            os.rename(path, reject_path)
        except OSError as error:
            log.error('error moving %s to reject location: %s', path, error)

    def _create_reason_file(self, text):
        filename = os.path.basename(self.changes_file_path)
        reason_name = filename[:filename.rfind('.')]
        reject_path = os.path.abspath(os.path.join(os.path.dirname(self.changes_file_path), '../REJECT'))
        with open('{}/{}.reason'.format(reject_path, reason_name), 'w+') as f:
            f.write(text)

    def _reject_package(self, changes_name, repo, changed_by, reason, err_lvl):
        if err_lvl == 0:
            log.info(reason + ' Rejecting.')
        if err_lvl == 1:
            log.warning(reason + ' Rejecting.')
        if err_lvl == 2:
            log.error(reason + ' Rejecting.')
        if err_lvl == 3:
            log.critical(reason + ' Rejecting.')

        self._create_reason_file(reason)
        pkg_name = os.path.basename(changes_name)
        if not self.notification:
            self.notification = notifications.reject_changes(
                repo,
                pkg_name,
                time.time(),
                reason,
                changed_by,
            )
        self.upload_status = UploadStatus.REJECT_NEEDED
        duploader_rejected_package_count.put_value(1)

    def guess_author_email(self):
        result = set()
        email_re = re.compile(r'.*<(?P<mail>[a-zA-Z0-9.-_]+@yandex-team\.ru)>.*')
        with open(self.changes_file_path, 'r') as f:
            for line in f:
                m = email_re.match(line.strip())
                if m is not None:
                    r = str(m.groupdict()['mail'])
                    result.add(r)
        result = list(result)
        log.info('guessed changes author emails: %s', result)
        return result

    @callable_timing_hram(duploader_package_upload_timing_hgram)
    def _processChangesFile(self, event):
        audit_meta = []
        log.info("Processing .changes file %s", event.pathname)
        log.debug("Sleeping 3s to ensure all changes are arrived")
        time.sleep(3)
        log.debug("Finished sleep")
        self.incoming_files = [event.pathname]
        changes = change_file.ChangeFile()
        log.debug("Loading ChangeFile from: %s", event.pathname)
        all_deb_files = set()
        try:
            changes.load_from_file(event.pathname)
            all_deb_files = {os.path.join(event.path, f[2]) for f in changes.get_files()}
            changes.filename = event.pathname
        except Exception as error:
            log.critical('cannot load changes: "%s", "%s"', event.pathname, error)
            _, _, tb = sys.exc_info()
            changes_error = ''.join(traceback.format_exception(type(error), error, tb))
            log.critical(changes_error)
            self.notification = notifications.malformed_changes(
                self.repo,
                os.path.basename(self.changes_file_path),
                changes_error,
                time.time(),
                self.guess_author_email()
            )
            self.upload_status = UploadStatus.CLEANUP_NEEDED
            return
        duploader_total_package_count.put_value(1)
        # .changes file contatins all incoming files and its checksums, so
        # check if all files are available of wait for them
        for filename in all_deb_files:
            attempts = 0
            timeout = infra.dist.cacus.lib.common.config['duploader_daemon']['incoming_wait_timeout']
            while True:
                if not os.path.exists(filename):
                    log.warning('file %s absent waiting for it %s sec...', filename, timeout)
                    time.sleep(float(timeout))
                    attempts += 1
                else:
                    self.incoming_files.append(filename)
                    break
                if attempts > 5:
                    raise DebWaitAttemptsExceeded(
                        'more than 5 attemps done waiting incoming files for {}. giving up.'.format(
                            self.changes_file_path))
        log.debug("Changes are ok. Moving files to work dir")
        all_deb_files = self._move_to_workdir(all_deb_files)
        log.debug('reloading changes from work dir')
        work_changes = self._move_to_workdir([event.pathname])[0]
        changes.load_from_file(work_changes)
        changes.filename = work_changes
        event.pathname = work_changes
        event.path = os.path.dirname(work_changes)
        self.incoming_files = [event.pathname]
        # .changes file contatins all incoming files and its checksums, so
        # check if all files are available of wait for them
        for filename in all_deb_files:
            if not os.path.exists(filename):
                log.error('file: %s is absent in work dir. Giving up.', filename)
                return
            else:
                self.incoming_files.append(filename)
        log.debug('reloaded changes')
        self.parsed_chages = changes
        changed_by_user = pwd.getpwuid(os.stat(event.pathname).st_uid).pw_name
        audit_meta.append({
            'event': 'upload',
            'user': changed_by_user,
            'timestamp': datetime.datetime.utcnow()
        })
        changed_by = changes['changed-by'] if 'changed-by' in changes else None

        default_env = 'unstable'
        rjct_pkg = partial(
            self._reject_package, changes.filename, self.repo, changed_by)
        if self.repo_conf.consider_distribution_string:
            if 'distribution' in changes:
                default_env = changes['distribution']
        skip_upstraem_check = False
        if not self.repo_conf.skip_gpg:
            try:
                signer = self._gpg_check(changes.filename, self.repo)
            except RuntimeError as e:
                msg = "{} verification failed: {}".format(event.pathname, e)
                rjct_pkg(msg, 1)
                return

            is_barmaleys = map(lambda x: x in signer, legacy.barmaley)
            skip_upstraem_check = bool(filter(lambda x: x, is_barmaleys))

            msg = "{}: signed by {}: OK".format(
                event.pathname, signer)
            log.info(msg)

        duplicate_restricted_repos = map(
            lambda x: 'yandex-{}'.format(x), self.upstream_dists_getter.get_dists()
        )
        if self.repo in duplicate_restricted_repos and not skip_upstraem_check:
            deb_in_upstream = self._check_for_upstream_debs(all_deb_files)
            if deb_in_upstream:
                msg = 'Upstream repsitory "{}" already contains package' \
                      ' with name {}.'.format(
                    self.repo.replace('yandex-', ''),
                    deb_in_upstream
                )
                rjct_pkg(msg, 1)
                return
        try:
            changes.verify(event.path)
            md5_sums = {os.path.join(event.path, f[2]): (f[0], f[1]) for f in changes.get_files()}
            st = os.stat(event.pathname)
            md5_sums[event.pathname] = (self._get_md5_sum(event.pathname), int(st.st_size))
        except change_file.ChangeFileException as e:
            msg = "Checksum verification failed: {}".format(e)
            rjct_pkg(msg, 1)
            return
        else:
            pkg_name_and_ver = '{}_{}'.format(
                changes['source'], changes['version']
            )
            msg = "{}: sign: OK, checksums: OK, uploading to repo '{}'," \
                  " environment '{}'".format(pkg_name_and_ver, self.repo, default_env)
            log.info(msg)
            n_try = 0
            while n_try < 6:
                try:
                    self.notification, err = repo_manage.upload_package(
                        self.repo,
                        default_env,
                        self.incoming_files,
                        changes=changes,
                        skipUpdateMeta=True,
                        audit_meta=audit_meta,
                        checksums=md5_sums
                    )
                    if err:
                        if not err.retryable:
                            msg = 'Error uploading {} package! Error: {}'.format(
                                pkg_name_and_ver, err
                            )
                            rjct_pkg(msg, 3)
                            return
                        msg = 'Retrying to upload package {}'.format(
                            pkg_name_and_ver
                        )
                        n_try += 1
                        log.info(msg)
                        sleep_time = randint(3 ** n_try / 2, 3 ** n_try)
                        log.warning(
                            'Next attemp to upload will be '
                            'performed in {} seconds'.format(sleep_time)
                        )
                        time.sleep(sleep_time)
                    else:
                        break
                except Exception as e:
                    msg = 'Error uploading {} package! Error: {}'.format(
                        pkg_name_and_ver, e
                    )
                    rjct_pkg(msg, 3)
                    log.exception(msg)
                    return
            else:
                msg = 'Reached max retrys count, while trying ' \
                      'to upload package {}'.format(pkg_name_and_ver)
                rjct_pkg(msg, 3)
                return
            # set status for cleanup
            if self.mode == 'compatibility':
                self.upload_status = UploadStatus.MOVE_TO_DINSTALL_NEEDED
            else:
                self.upload_status = UploadStatus.CLEANUP_NEEDED
            if self.notification is not None:
                self.notification.send()
            duploader_uploaded_package_count.put_value(1)
            log.info("Successfully duploaded {}".format(pkg_name_and_ver))
