from __future__ import absolute_import
import time
import os
import logging
from concurrent.futures import ThreadPoolExecutor
import infra.dist.cacus.lib.common
import infra.dist.cacus.lib.daemon.duploader.legacy
from infra.dist.cacus.lib.daemon.duploader.changes_worker import ChangesWorker
from infra.dist.cacus.lib.daemon.duploader.directory_maker_worker import DirectoryMakerWorker
from infra.dist.cacus.lib.daemon.duploader.incoming_cleanup_worker import IncomingCleanupWorker
from infra.dist.cacus.lib.daemon.duploader.workdir_cleanup_worker import WorkdirCleanupWorker
from infra.dist.cacus.lib.daemon.duploader.rejects_cleanup_worker import RejectsCleanupWorker
from infra.dist.cacus.lib.daemon.duploader.stats import dump_stats
from infra.dist.cacus.lib.daemon.duploader import repo_queue
from infra.dist.cacus.lib.dbal import package_repository

log = logging.getLogger(__name__)


class PeriodicFSWatcher:
    INCOMING_PATH_TEMPLATE = '{}/{}/mini-dinstall/incoming'
    REPO_PATH_TEMPLATE = '{}/{}'
    CHANGES_SUFFIX = '.changes'
    CHANGES_TEMPLATE = '{}/{}'
    REPOS_CONFIG_TTL = 30

    def __init__(self, root_path='/repo', scan_interval=5, max_workers=20, queue_config=None):
        self.root_path = root_path
        self.scan_interval = scan_interval
        self.max_workers = max_workers
        self.changes_thread_pool = ThreadPoolExecutor(max_workers=self.max_workers)
        self.maintenance_thread_pool = ThreadPoolExecutor(max_workers=5)
        # self.results = []
        # self.submitted_changes = {}
        self.repos_config_timestamp = 0
        self.repos_config = {}
        self.upstream_dists_getter = infra.dist.cacus.lib.common.UpstreamDistsGetter()
        self.q = repo_queue.RepoQueue(self.changes_thread_pool, config=queue_config)

    def run(self):
        # first run initialization
        if not infra.dist.cacus.lib.daemon.duploader.legacy.keep_gpg_cache_fresh():
            log.error('error updating cache. restarting cycle...')
            exit(-1)
        if not self.update_repos_config():
            log.error('error updating repo config. restarting cycle...')
            exit(-1)
        log.info('trying to create directories')
        try:
            DirectoryMakerWorker(self.repos_config).run()
        except Exception as error:
            log.exception('error creating directories on first run: {}'.format(error))
            exit(-1)
        while True:
            try:
                log.debug('starting repo scan')
                if not infra.dist.cacus.lib.daemon.duploader.legacy.keep_gpg_cache_fresh():
                    log.error('error updating cache. restarting cycle...')
                    continue
                if not self.update_repos_config():
                    log.error('error updating repo config. restarting cycle...')
                    continue
                self.maintenance_thread_pool.submit(DirectoryMakerWorker(self.repos_config))
                self.maintenance_thread_pool.submit(self.upstream_dists_getter.get_dists)
                self.maintenance_thread_pool.submit(RejectsCleanupWorker())
                self.maintenance_thread_pool.submit(WorkdirCleanupWorker())
                self.maintenance_thread_pool.submit(IncomingCleanupWorker())
                for d in os.listdir(self.root_path):
                    log.debug('found path: {}'.format(d))
                    incoming_path = self.incoming_path(d)
                    if self._isdir(incoming_path):
                        log.debug('path: {} contains incoming dir. scanning it'.format(incoming_path))
                        for f in os.listdir(incoming_path):
                            log.debug('found file: {}'.format(f))
                            changes_file = self.CHANGES_TEMPLATE.format(incoming_path, f)
                            if self._ischanges(changes_file):
                                if not self.q.has_job(d, changes_file):
                                    log.info('found changes file: {}'.format(changes_file))
                                    worker = ChangesWorker(changes_file, self.upstream_dists_getter, self.repos_config)
                                    self.q.put(d, changes_file, worker)
                                else:
                                    self.q.rebalance()
                dump_stats(infra.dist.cacus.lib.common.config['duploader_daemon']['stats_file_path'])
                self.q.rebalance()
                time.sleep(self.scan_interval)
            except Exception as error:
                log.exception('got error running periodic_fs_watcher: {}'.format(error))
                continue

    def incoming_path(self, repo):
        return self.INCOMING_PATH_TEMPLATE.format(self.root_path, repo)

    def repo_path(self, repo):
        return self.REPO_PATH_TEMPLATE.format(self.root_path, repo)

    def update_repos_config(self):
        try:
            if time.time() - self.repos_config_timestamp > self.REPOS_CONFIG_TTL:
                self.repos_config = package_repository.Config.all()
        except Exception as error:
            log.exception('error updating repos config: {}'.format(error))
            return False
        return True

    @staticmethod
    def _isdir(path):
        return os.path.isdir(path)

    @staticmethod
    def _isfile(path):
        return os.path.isfile(path)

    @staticmethod
    def _ischanges(path):
        return PeriodicFSWatcher._isfile(path) and path.endswith(PeriodicFSWatcher.CHANGES_SUFFIX)
