import logging
import collections
from infra.dist.cacus.lib import constants


log = logging.getLogger(__name__)


class JobQueue(object):
    def __init__(self, repo, tpe, max_parallel_jobs=3):
        self.repo = repo
        self._tpe = tpe
        self.max_parallel_jobs = max_parallel_jobs
        self.rq = {}
        self.sq = collections.OrderedDict()

    def __contains__(self, changes):
        return changes in self.rq or changes in self.sq

    def rebalance(self):
        log.debug("Repo {} queue before rebalance: rq: {}, sq: {}".format(self.repo,
                                                                          ', '.join(self.rq.keys() or ('<empty>',)),
                                                                          ', '.join(self.sq.keys() or ('<empty>',))))
        done = []
        for c, f in self.rq.iteritems():
            if f.done():
                log.info("Done processing changes: {}".format(c))
                done.append(c)
        for i in done:
            del self.rq[i]
        while len(self.rq) < self.max_parallel_jobs and len(self.sq) > 0:
            c, w = self.sq.popitem(False)
            self.rq[c] = self._tpe.submit(w, tuple(), {})
        log.debug("Repo {} queue after rebalance: rq: {}, sq: {}".format(self.repo,
                                                                          ', '.join(self.rq.keys() or ('<empty>',)),
                                                                          ', '.join(self.sq.keys() or ('<empty>',))))

    def put(self, changes, worker):
        self.sq[changes] = worker
        self.rebalance()


class RepoQueue(object):
    def __init__(self, tpe, default_jobs_per_repo=3, config=None):
        if config:
            self.config = config
        else:
            self.config = {constants.DEFAULT_JOBS_PER_REPO: default_jobs_per_repo}
        self._tpe = tpe
        self.q = {}

    def has_job(self, repo, changes):
        return repo in self.q and changes in self.q[repo]

    def _make_queue(self, repo):
        run_limit = self.config.get(repo, self.config[constants.DEFAULT_JOBS_PER_REPO])
        log.debug("Creating queue for repo: {} with run limit: {}", repo, run_limit)
        return JobQueue(repo, self._tpe, max_parallel_jobs=run_limit)

    def put(self, repo, changes, worker):
        if repo not in self.q:
            self.q[repo] = self._make_queue(repo)
        if not self.has_job(repo, changes):
            self.q[repo].put(changes, worker)

    def rebalance(self):
        for _, q in self.q.iteritems():
            q.rebalance()
