import logging
import os
import time
import traceback
import sys
from infra.dist.cacus.lib.daemon.duploader.worker import Worker
from infra.dist.cacus.lib.daemon.duploader.stats import duploader_stale_work_removed_count

log = logging.getLogger(__name__)


class WorkdirCleanupWorker(Worker):
    WORK_DIR_TEMPLATE = '{}/{}/mini-dinstall/work'

    def __init__(self, root_path='/opt/repo', max_age=2592000):
        self.root_path = root_path
        self.cleanup_list = []
        self.max_age = max_age

    def run(self):
        self.walk_tree()

    def cleanup(self):
        for p in self.cleanup_list:
            try:
                log.debug('unlinking file: %s', p)
                os.unlink(p)
                duploader_stale_work_removed_count.put_value(1)
            except OSError as error:
                log.error('cannot unlink file: %s reason: %s', p, error)
                _, _, tb = sys.exc_info()
                tb = traceback.format_exception(type(error), error, tb)
                log.error(''.join(tb))
                continue

    def walk_tree(self):
        for d in os.listdir(self.root_path):
            log.debug('scanning possible repo: %s', d)
            dir_path = self.WORK_DIR_TEMPLATE.format(self.root_path, d)
            if os.path.isdir(dir_path):
                log.debug('scanning directory: %s for files to remove', dir_path)
                for f in os.listdir(dir_path):
                    file_path = os.path.join(dir_path, f)
                    if self.remove_file(file_path):
                        log.debug('scheduling file: %s to remove', file_path)
                        self.cleanup_list.append(file_path)

    def remove_file(self, path):
        log.debug('checking file: %s', path)
        if os.path.isfile(path):
            st = os.stat(path)
            if time.time() - st.st_atime >= self.max_age:
                log.debug('file %s: is older than %s will schedule to remove', path, self.max_age)
                return True
            else:
                log.debug('file %s: is NOT older than %s won\'t schedule to remove', path, self.max_age)
                return False
        else:
            return False
