#! /usr/bin/env python

import hashlib
import json
import logging
import os
import re
import subprocess
import time
from tempfile import mkstemp

from kazoo.exceptions import LockTimeout

import infra.dist.cacus.lib.loader
from infra.dist.cacus.lib import common
from infra.dist.cacus.lib.daemon.gpg.stats import stats
from infra.dist.cacus.lib.dbal import errors
from infra.dist.cacus.lib.dbal import keyring as db_keyring
from infra.dist.cacus.lib.utils.gpg import get_gpg_keyrings

log = logging.getLogger(__name__)
size_diff_threshold = 40


class GPGKeyringHasStrangeSizeDiff(Exception):
    pass


def dump_stats(file_path):
    metric_list = []
    for k, v in stats.items():
        metric_list.append(
            [
                k,
                v
            ]
        )
    with open(file_path, 'w') as f:
        json.dump(metric_list, f)


def generate_keyring_shasum(keyring):
    fd, tmppath = mkstemp()
    tmpfile = open(tmppath, 'wb+')
    tmpfile.write(keyring)
    tmpfile.close()
    os.close(fd)
    for i in range(3):
        try:
            with common.timeout(seconds=30):
                p = subprocess.Popen(
                    [
                        'gpg',
                        '--no-default-keyring',
                        '--keyring',
                        tmppath,
                        '--fingerprint'],
                    stdout=subprocess.PIPE
                )
                gpg_out, stderrdata = p.communicate()
        except common.TimeoutError:
            pass
    os.unlink(tmppath)
    fingerprint_pattern = re.compile(
        '((?:[0-9A-F]{4} ){5} (?:[0-9A-F]{4} ){4}[0-9A-F]{4})'
    )
    log.debug('gpg output: {}'.format(gpg_out))
    fingerprints = list(re.findall(fingerprint_pattern, gpg_out))
    fingerprints.sort()

    m = hashlib.sha256()
    for fingerprint in fingerprints:
        m.update(fingerprint)

    return m.hexdigest()


def extract_group_from_metadata(group, metadata):
    for keyring_meta in metadata:
        if keyring_meta.group == group:
            return keyring_meta
    return None


def update_keyring_for_group(group, new_keyring, old_metadata, force_update=False, keyring_storage=db_keyring.default_storage):
    modified = False

    keyring_sha256 = generate_keyring_shasum(new_keyring)
    log.debug('updating keyring: {} sha256: {}'.format(group, keyring_sha256))
    meta_group = extract_group_from_metadata(group, old_metadata)
    if not meta_group:
        log.debug('keyring {} not found in DB'.format(group))
        log.debug('keyring {} modified'.format(group))
        modified = True
    else:
        log.debug(meta_group)
        log.debug('found keyring: {} sha256: {} in DB'.format(group, meta_group.sha256))
        if keyring_sha256 != meta_group.sha256:
            modified = True
            log.debug('keyring {} modified'.format(group))
        else:
            log.debug('new keyring and DB record hashes are equal trying to check MDS content')
            response = infra.dist.cacus.lib.loader.get_plugin('storage').get(meta_group.storage_key)
            mds_keyring_sha256 = generate_keyring_shasum(response.content)
            log.debug('keyring MDS: {} sha256: {}'.format(group, mds_keyring_sha256))
            if mds_keyring_sha256 != keyring_sha256:
                modified = True
                log.debug('keyring {} modified: MDS sha256 differs DB sha256'.format(group))
            else:
                log.debug('keyring {} DB record consistent')
        len_diff = abs(len(new_keyring) - int(meta_group.size))
        size_diff = (len_diff * 100) / int(meta_group.size)
        if size_diff > size_diff_threshold and not force_update:
            raise GPGKeyringHasStrangeSizeDiff()
        if size_diff > size_diff_threshold and force_update:
            msg = 'New keyring differs with old for more, ' \
                  'than {} percents! Enforcing generation ' \
                  'for keyring: {}.gpg'.format(
                size_diff_threshold,
                group
            )
            log.warning(msg)

    if modified:
        keyring_file = common.myStringIO()
        keyring_file.write(new_keyring)
        key_name = '__gpg__/{}.gpg_{}'.format(group, keyring_sha256)
        storage_key = infra.dist.cacus.lib.loader.get_plugin('storage').put(
            key_name,
            file=keyring_file
        )
        k = db_keyring.Keyring(group, 'group', storage_key, keyring_sha256, len(new_keyring))
        try:
            keyring_storage.save(k)
        except errors.KeyringNotSaved:
            return False
        return True


def start_gpg_daemon(force_update=False, keyring_storage=db_keyring.default_storage):
    lock_factory = common.ZKLockFactory(logger=log)
    while True:
        zk_lock = lock_factory.get_lock('gpg_deamon', timeout=5)
        keyrings = get_gpg_keyrings()
        log.debug('got {} keyrings from get_gpg_keyrings()'.format(len(keyrings)))
        for k, v in keyrings.iteritems():
            log.debug('keyring: {} len: {}'.format(k, len(v)))
        try:
            log.debug('trying to get gpg-daemon lock')
            with zk_lock:
                log.debug('trying to get gpg keyrings from DB')
                keyrings_meta = keyring_storage.find(type_='group')
                log.debug('got {} keyrings from DB'.format(len(keyrings_meta)))
                log.debug('finally fetched keyrings from staff and DB')
                for group, keyring in keyrings.iteritems():
                    try:
                        log.debug('trying to update keyring for group: {}'.format(group))
                        update_keyring_for_group(
                            group, keyring,
                            keyrings_meta,
                            force_update=force_update,
                            keyring_storage=keyring_storage
                        )
                    except GPGKeyringHasStrangeSizeDiff:
                        msg = 'New keyring differs with old for more, ' \
                              'than {} percents! Skipping generation ' \
                              'for keyring: {}.gpg'.format(
                            size_diff_threshold,
                            group
                        )
                        log.critical(msg)
        except LockTimeout:
            log.warn('failed to get gpg-daemon lock')
            continue
        dump_stats(common.config['gpg_daemon']['stats_file_path'])
        time.sleep(300)

# {
#     "type": "group"
#     "center_group":
#     "storage_key":
#     "sha265":
#     "size":
# }
# {
#      "type": "meta"
#      "last_modified":
# }

# 1 pospat' 2 sekundy
# 2 posmotret' vremja poslednej generacii
# 3 esli data poslednego obnovlenija starshe 5 minut, nagenerit' vse brelki
# 4 posmotret' izmenitas' li kontrol'naja summa hotja by dlja odnog iz brelkov
# 5 esli da, to posmotret' na izmeneni razmera
# 6 esli s razmerom vse ok, to poprobovat' zahvatit' lok v zukipere
# 7 zaaploadit' izmenivshiesja brelki v MDS
# 8 poapdejtit' tablicu v BD
# 9 otpustit' log v zukipere
