from __future__ import absolute_import

import logging
import time

import infra.dist.cacus.lib.common
from infra.dist.cacus.lib.dbal import mongo_connection
from infra.dist.cacus.lib.stats.histogram import Histogram
from infra.dist.cacus.lib.stats.value import Value
from infra.dist.cacus.lib.utils.push_metrics import push_metric

log = logging.getLogger(__name__)
raw_metrics = {
    'connections.current': Value('mongo_connections_current_ammm'),
    'connections.available': Value('mongo_connections_available_ammm'),

    'opcounters.getmore': Value('mongo_op_getmore_summ'),
    'opcounters.insert': Value('mongo_op_insert_summ'),
    'opcounters.update': Value('mongo_op_update_summ'),
    'opcounters.delete': Value('mongo_op_delete_summ'),
    'opcounters.query': Value('mongo_op_query_summ'),
    'opcounters.command': Value('mongo_op_command_summ'),

    'opcountersRepl.getmore': Value('mongo_rop_getmore_summ'),
    'opcountersRepl.insert': Value('mongo_rop_insert_summ'),
    'opcountersRepl.update': Value('mongo_rop_update_summ'),
    'opcountersRepl.delete': Value('mongo_rop_delete_summ'),
    'opcountersRepl.query': Value('mongo_rop_query_summ'),
    'opcountersRepl.command': Value('mongo_rop_command_summ'),
}
computed_metrics = {
    'mongo_op_summ': Value('mongo_op_summ'),
    'mongo_rop_summ': Value('mongo_rop_summ'),
    'mongo_repl_lag_ahhh': Histogram('mongo_repl_lag_ahhh')
}
MONITOR_CONN = 'monitordb'


def get_key(status, key):
    path = key.split('.')
    for comp in path:
        status = status[comp]
    return status


def update_measure(measure, value):
    log.debug('updating measure: {} with value: {}'.format(measure.name, value))
    measure.reset_measure(True)
    measure.start_measure()
    measure.put_value(value)
    measure.end_measure()


def main():
    metrics_url = infra.dist.cacus.lib.common.config['mongo_stats_daemon']['stats_push_url']
    db = mongo_connection.MongoDatabaseConnection.from_config(MONITOR_CONN, infra.dist.cacus.lib.common.config['metadb'][MONITOR_CONN], wrapper=None)
    while True:
        try:
            status = db.get().command('serverStatus')
            for key, value in raw_metrics.items():
                update_measure(value, get_key(status, key))
            update_measure(
                computed_metrics['mongo_op_summ'],
                sum((v.get_value() for v in filter(lambda x: 'mongo_op_' in x.name, raw_metrics.values())))
            )
            update_measure(
                computed_metrics['mongo_rop_summ'],
                sum((v.get_value() for v in filter(lambda x: 'mongo_rop_' in x.name, raw_metrics.values())))
            )
            try:
                update_measure(
                    computed_metrics['mongo_repl_lag_ahhh'],
                    get_key(status, 'repl.lastWrite.majorityOpTime.ts').time - get_key(status,
                                                                                       'repl.lastWrite.opTime.ts').time
                )
            except KeyError:
                pass
            for metric in raw_metrics.values():
                if metric.has_values():
                    push_metric(metrics_url, metric.to_json())
            for metric in computed_metrics.values():
                if metric.has_values():
                    push_metric(metrics_url, metric.to_json())
            time.sleep(2)
        except KeyboardInterrupt:
            log.warning('keyboard interrupt. exiting...')
            exit(13)
        except Exception as error:
            log.error('got unexpected error running mongo_stats-daemon')
            log.error('{}: {}'.format(type(error), error))
