import copy

from infra.dist.cacus.lib.dbal import errors
from infra.dist.cacus.lib.dbal import mongo_connection


class Package(object):
    def __init__(self, repo, env, source, version, sources, debs, dsc, audit_meta=None, recycle_after=None):
        self.recycle_after = recycle_after
        self._audit_meta = audit_meta
        self.dsc = dsc
        self._debs = debs
        self._sources = sources
        self.version = version
        self.source = source
        self.env = env
        self.repo = repo
        self._new_audit_meta = []
        self._audit_meta_changed = False
        self._new_debs = []
        self._debs_changed = False
        self._new_sources = []
        self._sources_changed = False

    @property
    def audit_meta(self):
        return self._audit_meta

    def append_audit(self, r):
        if self._audit_meta_changed:
            raise errors.TransactionConflict('cannot use append_audit() when audit_meta replaced')
        self._new_audit_meta.append(r)

    def extend_audit(self, r):
        if self._audit_meta_changed:
            raise errors.TransactionConflict('cannot use extend_audit() when audit_meta replaced')
        self._new_audit_meta.extend(r)

    @audit_meta.setter
    def audit_meta(self, r):
        if self._new_audit_meta:
            raise errors.TransactionConflict(
                'cannot use set_audit() when new audit_meta pending: {}'.format(self._new_audit_meta))
        self._audit_meta = r
        self._audit_meta_changed = True

    @property
    def debs(self):
        return self._debs

    def append_debs(self, r):
        if self._debs_changed:
            raise errors.TransactionConflict('cannot use append_debs() when debs replaced')
        self._new_debs.append(r)

    def extend_debs(self, r):
        if self._debs_changed:
            raise errors.TransactionConflict('cannot use extend_debs() when debs replaced')
        self._new_debs.extend(r)

    @debs.setter
    def debs(self, r):
        if self._new_debs:
            raise errors.TransactionConflict(
                'cannot use set_debs() when new debs pending: {}'.format(self._new_debs))
        self._debs = r
        self._debs_changed = True

    @property
    def sources(self):
        return self._sources

    def append_sources(self, r):
        if self._sources_changed:
            raise errors.TransactionConflict('cannot use append_sources() when sources replaced')
        self._new_sources.append(r)

    def extend_sources(self, r):
        if self._sources_changed:
            raise errors.TransactionConflict('cannot use extend_sources() when sources replaced')
        self._new_sources.extend(r)

    @sources.setter
    def sources(self, r):
        if self._new_sources:
            raise errors.TransactionConflict(
                'cannot use set_sources() when new sources pending: {}'.format(self._new_sources))
        self._sources = r
        self._sources_changed = True

    @classmethod
    def empty(cls, repo):
        return cls(repo, env=None, source=None, version=None, sources=[], debs=[], dsc={}, audit_meta=[],
                   recycle_after=None)

    @classmethod
    def from_dict(cls, repo, d):
        return cls(
            repo,
            d['environment'],
            d['Source'],
            d['Version'],
            d.get('sources') or [],
            d.get('debs') or [],
            d.get('dsc') or {},
            d.get('audit_meta') or [],
            d.get('recycle_after')
        )

    @staticmethod
    def _drop_empty(d):
        """
        delete empty empty values, dicts and lists
        """
        for k in d.keys():
            if isinstance(d[k], dict):
                Package._drop_empty(d[k])
            if not d[k] and (isinstance(d[k], dict) or isinstance(d[k], list)) or d[k] is None:
                del d[k]

    def _mongo_update(self):
        """
        Build upsert query dict
        """
        d = {
            '$set': {
                'environment': self.env,
                'Source': self.source,
                'Version': self.version,
                'dsc': self.dsc,
                'recycle_after': self.recycle_after,
            },
            '$push': {}
        }
        if self._new_audit_meta:
            d['$push']['audit_meta'] = {'$each': self._new_audit_meta}
        else:
            d['$set']['audit_meta'] = self.audit_meta
        if self._new_debs:
            d['$push']['debs'] = {'$each': self._new_debs}
        else:
            d['$set']['debs'] = self.debs
        if self._new_sources:
            d['$push']['sources'] = {'$each': self._new_sources}
        else:
            d['$set']['sources'] = self.sources
        self._drop_empty(d)
        return d

    @staticmethod
    def _find_query(env=None, source=None, version=None, recycle_after=None, arch=None, dsc=None, package=None):
        query = {}
        if env:
            query['environment'] = env
        if source:
            query['Source'] = source
        if version:
            query['Version'] = version
        if recycle_after:
            query['recycle_after'] = recycle_after
        if arch:
            query['debs.Architecture'] = arch
        if dsc:
            query['dsc'] = dsc
        if package:
            pq = copy.deepcopy(query)
            pq.pop('Source')
            pq['debs'] = {
                '$elemMatch': {
                    'Package': package
                }
            }
            query = {
                '$or': [
                    query,
                    pq
                ]
            }
        return query

    @classmethod
    def find(cls, repo, env=None, source=None, version=None, recycle_after=None, arch=None, dsc=None, package=None):
        query = cls._find_query(env, source, version, recycle_after, arch, dsc, package)
        results = []
        for i in mongo_connection.repos()[repo].find(query):
            results.append(Package.from_dict(repo, i))
        return results

    @classmethod
    def find_one(cls, repo, env=None, source=None, version=None, recycle_after=None, arch=None, dsc=None, package=None):
        query = cls._find_query(env, source, version, recycle_after, arch, dsc, package)
        r = mongo_connection.repos()[repo].find_one(query)
        if r:
            return cls.from_dict(repo, r)
        else:
            return None

    @classmethod
    def delete_many(cls, repo, env=None, source=None, version=None, recycle_after=None, arch=None, dsc=None,
                    package=None):
        query = cls._find_query(env, source, version, recycle_after, arch, dsc, package)
        return mongo_connection.repos()[repo].delete_many(query)

    def delete(self):
        query = {
            'Source': self.source,
            'Version': self.version,
        }
        return mongo_connection.repos()[self.repo].delete_one(query)

    def save(self):
        query = {
            'Source': self.source,
            'Version': self.version,
        }
        return mongo_connection.repos()[self.repo].update_one(query, self._mongo_update(), upsert=True)

    @classmethod
    def from_package(cls, p):
        np = cls(repo=p.repo, env=p.env, source=p.source, version=p.version, sources=None, debs=None, dsc=None,
                 audit_meta=None, recycle_after=p.recycle_after)
        ndebs = copy.deepcopy(p.debs)
        np.debs = ndebs
        nsources = copy.deepcopy(p.sources)
        np.sources = nsources
        naudit = copy.deepcopy(p.audit_meta)
        np.audit_meta = naudit
        np.dsc = copy.deepcopy(p.dsc)
        return np
