import datetime

from infra.dist.cacus.lib.dbal import mongo_connection
from infra.dist.cacus.lib.dbal import package_index_history


class PackageIndexStore(object):
    def __init__(self, db_fun=mongo_connection.cacus):
        self.db = db_fun

    @staticmethod
    def _find_query(env, arch, dirty_bit=None):
        q = {'environment': env, 'architecture': arch}
        if dirty_bit is not None:
            q['dirty_bit'] = dirty_bit
        return q

    def all(self, repo):
        return [PackageIndex.from_dict(repo, x) for x in self.db()[repo].find()]

    def find_one(self, repo, env, arch, dirty_bit=None):
        r = self.db()[repo].find_one(self._find_query(env, arch, dirty_bit))
        if r:
            return PackageIndex.from_dict(repo, r)
        else:
            return None

    def set_dirty(self, repo, env, arch):
        return self.db()[repo].update_one(
            self._find_query(env, arch),
            {
                '$set': {
                    'dirty_bit': True,
                    'dirty_bit_set_at': datetime.datetime.utcnow()
                }
            },
            upsert=True,
        )

    def set_clean(self, repo, env, arch):
        return self.db()[repo].update_one(
            self._find_query(env, arch),
            {
                '$set': {
                    'dirty_bit': False,
                    'dirty_bit_set_at': datetime.datetime.utcnow(),
                    'force_index': False,
                }
            },
        )

    def save(self, idx):
        """
        Save PackageIndex to DB and maintain history
        """
        # save current index to history if it exists
        # the order save -> update is mandatory to avoid deleting current keys from MDS
        prev_idx = self.find_one(idx.repo, idx.env, idx.arch)
        if prev_idx:
            package_index_history.default_store.put(prev_idx)
        return self.db()[idx.repo].update_one(self._find_query(idx.env, idx.arch),
                                              {'$set': idx.to_dict()}, upsert=True)

    def delete(self, idx):
        return self.db()[idx.repo].delete_one(self._find_query(idx.env, idx.arch))

    def drop(self, repo):
        return self.db()[repo].drop()


class PackageIndex(object):
    def __init__(self, repo, arch=None, env=None, size=None, plain=None, gzipped=None,
                 sha1=None, release_gpg=None, bzipped=None, sha256=None, release_file=None, lastupdated=None,
                 md5=None, dirty_bit=None, dirty_bit_set_at=None, force_index=None, in_release_file=None,
                 release_gpg_by_hash=None, gzipped_sha256=None, bzipped_sha256=None, in_release_gpg_by_hash=None,
                 release_by_hash=None, plain_sha256=None):
        self.repo = repo
        self.in_release_gpg_by_hash = in_release_gpg_by_hash
        self.in_release_file = in_release_file
        self.lastupdated = lastupdated
        self.release_file = release_file
        self.bzipped_sha256 = bzipped_sha256
        self.force_index = force_index
        self.sha256 = sha256
        self.bzipped = bzipped
        self.dirty_bit = dirty_bit
        self.release_gpg = release_gpg
        self.sha1 = sha1
        self.md5 = md5
        self.dirty_bit_set_at = dirty_bit_set_at
        self.gzipped_sha256 = gzipped_sha256
        self.release_gpg_by_hash = release_gpg_by_hash
        self.plain_sha256 = plain_sha256
        self.release_by_hash = release_by_hash
        self.gzipped = gzipped
        self.plain = plain
        self.size = size
        self.env = env
        self.arch = arch

    @classmethod
    def from_dict(cls, repo, d):
        plain_key, gzipped_key, bzipped_key = cls._keys_for_arch(d['architecture'])
        return cls(
            repo,
            in_release_gpg_by_hash=d.get('in_release_gpg_by_hash'),
            in_release_file=d.get('in_release_file'),
            lastupdated=d.get('lastupdated'),
            release_file=d.get('release_file'),
            bzipped_sha256=d.get('bzipped_sha256'),
            force_index=d.get('force_index'),
            sha256=d.get('sha256'),
            bzipped=d.get(bzipped_key),
            dirty_bit=d.get('dirty_bit'),
            release_gpg=d.get('release_gpg'),
            sha1=d.get('sha1'),
            md5=d.get('md5'),
            dirty_bit_set_at=d.get('dirty_bit_set_at'),
            gzipped_sha256=d.get('gzipped_sha256'),
            release_gpg_by_hash=d.get('release_gpg_by_hash'),
            plain_sha256=d.get('plain_sha256'),
            release_by_hash=d.get('release_by_hash'),
            gzipped=d.get(gzipped_key),
            plain=d.get(plain_key),
            size=d.get('size'),
            env=d.get('environment'),
            arch=d.get('architecture')
        )

    @classmethod
    def from_index(cls, idx):
        return cls(
            idx.repo,
            in_release_gpg_by_hash=idx.in_release_gpg_by_hash,
            in_release_file=idx.in_release_file,
            lastupdated=idx.lastupdated,
            release_file=idx.release_file,
            bzipped_sha256=idx.bzipped_sha256,
            force_index=idx.force_index,
            sha256=idx.sha256,
            bzipped=idx.bzipped,
            dirty_bit=idx.dirty_bit,
            release_gpg=idx.release_gpg,
            sha1=idx.sha1,
            md5=idx.md5,
            dirty_bit_set_at=idx.dirty_bit_set_at,
            gzipped_sha256=idx.gzipped_sha256,
            release_gpg_by_hash=idx.release_gpg_by_hash,
            plain_sha256=idx.plain_sha256,
            release_by_hash=idx.release_by_hash,
            gzipped=idx.gzipped,
            plain=idx.plain,
            size=idx.size,
            env=idx.env,
            arch=idx.arch,
        )

    @staticmethod
    def _keys_for_arch(arch):
        """
        returns (plain_key, gzipped_key, bzipped_key)
        """
        if arch == 'source':
            return 'sources_file', 'gziped_sources', 'bziped_sources'
        else:
            return 'packages_file', 'gziped_packages', 'bziped_packages'

    def to_dict(self):
        plain_key, gzipped_key, bzipped_key = self._keys_for_arch(self.arch)
        return {
            'in_release_gpg_by_hash': self.in_release_gpg_by_hash,
            'in_release_file': self.in_release_file,
            'lastupdated': self.lastupdated,
            'release_file': self.release_file,
            'bzipped_sha256': self.bzipped_sha256,
            'force_index': self.force_index,
            'sha256': self.sha256,
            bzipped_key: self.bzipped,
            'dirty_bit': self.dirty_bit,
            'release_gpg': self.release_gpg,
            'sha1': self.sha1,
            'md5': self.md5,
            'dirty_bit_set_at': self.dirty_bit_set_at,
            'gzipped_sha256': self.gzipped_sha256,
            'release_gpg_by_hash': self.release_gpg_by_hash,
            'plain_sha256': self.plain_sha256,
            'release_by_hash': self.release_by_hash,
            gzipped_key: self.gzipped,
            plain_key: self.plain,
            'size': self.size,
            'environment': self.env,
            'architecture': self.arch,
        }


default_store = PackageIndexStore()
