import datetime
import logging

import pymongo

from infra.dist.cacus.lib import loader, constants
from infra.dist.cacus.lib.dbal import mongo_connection

DEFAULT_HISTORY_VALIDITY = 600
log = logging.getLogger(__name__)


def default_store_policy(conf, entry, ts):
    valid_delta = conf.get('policy', {}).get('max_age', DEFAULT_HISTORY_VALIDITY)
    valid_delta = conf.get('policy', {}).get('repos', {}).get(entry.repo, {}).get('max_age', valid_delta)
    entry.valid_before = ts + datetime.timedelta(seconds=valid_delta)


class PackageIndexHistoryStore(object):
    def __init__(self, db_fun=mongo_connection.history, db_conf_fun=mongo_connection.history_config,
                 policy_fun=default_store_policy):
        self.db = db_fun
        self.db_conf = db_conf_fun
        self.policy_fun = policy_fun

    def put(self, idx):
        """
        Puts PackageIndex to history DB and maintains history DB cleanup
        """
        history = PackageIndexHistoryEntry.from_index(idx)
        now = datetime.datetime.utcnow()
        # update history and run cleanup on success
        # order save -> cleanup is mandatory to avoid deleting current keys from MDS
        for i in history:
            self.policy_fun(self.db_conf(), i, now)
            self.save(i)
        self.cleanup(idx.repo, idx.env, idx.arch, (idx.bzipped_sha256, idx.gzipped_sha256, idx.plain_sha256))

    def _ensure_indexes(self, repo):
        self.db()[repo].create_index([
            ("environment", pymongo.ASCENDING),
            ("architecture", pymongo.ASCENDING),
        ],
            name="env-arch"
        )
        self.db()[repo].create_index([
            ("environment", pymongo.ASCENDING),
            ("architecture", pymongo.ASCENDING),
            ("sha256", pymongo.ASCENDING),
        ],
            name="env-arch-sha256"
        )
        self.db()[repo].create_index([
            ("environment", pymongo.ASCENDING),
            ("architecture", pymongo.ASCENDING),
            ("valid_before", pymongo.ASCENDING),
        ],
            name="env-arch-validity"
        )

    def save(self, entry):
        self._ensure_indexes(entry.repo)
        return self.db()[entry.repo].update_one(self._find_query(env=entry.env, arch=entry.arch, sha256=entry.sha256),
                                                {'$set': entry.to_dict()}, upsert=True)

    @staticmethod
    def _find_query(env=None, arch=None, valid_before=None, sha256=None):
        q = {}
        if env:
            q['environment'] = env
        if arch:
            q['architecture'] = arch
        if valid_before:
            q['valid_before'] = valid_before
        if sha256:
            q['sha256'] = sha256
        return q

    def cleanup(self, repo, env, arch, protected_hashes=None):
        """
        Finds stale history records for (repo, env, arch), deletes keys from MDS and deletes records from DB
        """
        before_now = {'$lt': datetime.datetime.utcnow()}
        invalid_entries = [
            PackageIndexHistoryEntry.from_dict(repo, x) for x in
            self.db()[repo].find(self._find_query(env=env, arch=arch, valid_before=before_now))
        ]
        protected_hashes = protected_hashes or tuple()
        for i in invalid_entries:
            # skip protected hashes deletion from MDS
            if i.sha256 in protected_hashes:
                log.debug("Skipped protected history index:{}, repo: {}, env: {}, arch: {}, storage_key: {}".format(
                    i.sha256, repo, env, arch, i.storage_key))
                continue
            if constants.MDS_EMPTY_KEY in i.storage_key:
                log.debug("Skipped empty history index:{}, repo: {}, env: {}, arch: {}, storage_key: {}".format(
                    i.sha256, repo, env, arch, i.storage_key))
                continue
            log.info("Cleaning expired history index:{}, repo: {}, env: {}, arch: {}".format(i.sha256, repo, env, arch))
            err = loader.get_plugin('storage').delete(i.storage_key)
            if err:
                log.error(
                    "Failed to cleanup expired history index:{}, repo: {}, env: {}, arch: {}: {}".format(i.sha256, repo,
                                                                                                         env, arch,
                                                                                                         err))
                return
        # we can delete protected hashes from history DB they will be restored on next put()
        self.db()[repo].delete_many(self._find_query(env=env, arch=arch, valid_before=before_now))


class PackageIndexHistoryEntry(object):
    def __init__(self, repo, env, arch, sha256, storage_key, valid_before, updated_at):
        self.repo = repo
        self.env = env
        self.arch = arch
        self.sha256 = sha256
        self.storage_key = storage_key
        self.valid_before = valid_before
        self.updated_at = updated_at

    @classmethod
    def from_dict(cls, repo, d):
        return cls(
            repo,
            d['environment'],
            d['architecture'],
            d['sha256'],
            d['storage_key'],
            d['valid_before'],
            d['updated_at'],
        )

    def to_dict(self):
        return {
            'environment': self.env,
            'architecture': self.arch,
            'sha256': self.sha256,
            'storage_key': self.storage_key,
            'valid_before': self.valid_before,
            'updated_at': self.updated_at,
        }

    @classmethod
    def from_index(cls, idx):
        rv = []
        if idx.plain_sha256:
            rv.append(cls(
                idx.repo,
                idx.env,
                idx.arch,
                idx.plain_sha256,
                idx.plain,
                None,
                idx.lastupdated,
            ))
        if idx.gzipped_sha256:
            rv.append(cls(
                idx.repo,
                idx.env,
                idx.arch,
                idx.gzipped_sha256,
                idx.gzipped,
                None,
                idx.lastupdated,
            ))
        if idx.bzipped_sha256:
            rv.append(cls(
                idx.repo,
                idx.env,
                idx.arch,
                idx.bzipped_sha256,
                idx.bzipped,
                None,
                idx.lastupdated,
            ))
        return rv


default_store = PackageIndexHistoryStore()
