import re
from pymongo import ASCENDING

from infra.dist.cacus.lib import constants
from infra.dist.cacus.lib.dbal import errors
from infra.dist.cacus.lib.dbal import mongo_connection


class Config(object):
    DEFAULT_REPO_DESCRIPTION = 'empty repo description'

    def __init__(self, repo, skip_gpg=False, repo_root=None, incoming_dir=None, description=None,
                 consider_distribution_string=False, loose_upload_checks=False, generate_torrent=False):
        self.generate_torrent = generate_torrent
        self.loose_upload_checks = loose_upload_checks
        self.consider_distribution_string = consider_distribution_string
        self.description = description
        self.incoming_dir = incoming_dir
        self.repo_root = repo_root
        self.repo = repo
        self.skip_gpg = skip_gpg

    @staticmethod
    def _repo_root(repo):
        return '/opt/repo/{}'.format(repo)

    @staticmethod
    def _incoming_dir(repo):
        '{}/mini-dinstall/incoming'.format(Config._repo_root(repo))

    @classmethod
    def default(cls, repo, description=DEFAULT_REPO_DESCRIPTION):
        return cls(repo, skip_gpg=False,
                   repo_root=cls._repo_root(repo),
                   incoming_dir=cls._incoming_dir(repo),
                   description=description,
                   consider_distribution_string=False,
                   loose_upload_checks=False,
                   generate_torrent=False)

    @classmethod
    def from_dict(cls, d):
        """
        dict should look like:
        {
            <repo>: {
                <p1>: <v1>,
                <p2>: <v2>,
                ...
                <pN>: <vN>,
            }
        }
        """
        repo = d.keys()[0]
        d = d[repo]
        return cls(repo,
                   skip_gpg=d.get('skip_gpg', False),
                   repo_root=d.get('repo_root', cls._repo_root(repo)),
                   incoming_dir=d.get('incoming_dir', cls._incoming_dir(repo)),
                   description=d.get('description', cls.DEFAULT_REPO_DESCRIPTION),
                   consider_distribution_string=d.get('consider_distribution_string', False),
                   loose_upload_checks=d.get('loose_upload_checks', False),
                   generate_torrent=d.get('generate_torrent', False),
                   )

    def to_dict(self):
        return {
            self.repo: {
                'skip_gpg': self.skip_gpg,
                'repo_root': self.repo_root,
                'incoming_dir': self.incoming_dir,
                'description': self.description,
                'consider_distribution_string': self.consider_distribution_string,
                'loose_upload_checks': self.loose_upload_checks,
                'generate_torrent': self.generate_torrent,
            }
        }

    @staticmethod
    def _from_db():
        r = list(mongo_connection.cacus()['__config__'].find())
        count = len(r)
        if count != 1:
            raise errors.MalformedConfigCollection('config collection has {} documents, but should have only 1'.format(count))
        return r[0]

    @classmethod
    def find(cls, repo):
        conf_dict = mongo_connection.cacus()['__config__'].find()[0]
        if repo not in conf_dict:
            raise errors.RepositoryNotFound('repository {} config not found'.format(repo))
        return cls.from_dict({repo: conf_dict[repo]})

    def save(self):
        r = mongo_connection.cacus()['__config__'].update_one({}, {'$set': self.to_dict()}, upsert=True)
        if r.upserted_id is None and r.matched_count != 1:
            raise errors.MalformedConfigCollection('config collection update should match exactly 1 document')

    @classmethod
    def all(cls):
        rv = {}
        r = Config._from_db()
        r.pop('_id')
        for k, v in r.items():
            rv[k] = cls.from_dict({k: v})
        return rv

    def delete(self):
        r = mongo_connection.cacus()['__config__'].update_one({}, {'$unset': {self.repo: 0}})
        if r.matched_count != 1:
            raise errors.MalformedConfigCollection('config collection update should match exactly 1 document')


def list_all():
    configs = Config.all()
    return configs.keys()


def add(repo, description, incoming_dir, repo_root, update_sources, update_packages):
    conf = Config(repo, description=description, incoming_dir=incoming_dir, repo_root=repo_root)
    conf.save()
    mongo_connection.repos()[repo].create_index('Source')
    mongo_connection.repos()[repo].create_index('debs.Package')
    mongo_connection.repos()[repo].create_index('debs.storage_key')
    mongo_connection.cacus()[repo].create_index('environment')
    mongo_connection.cacus()[repo].create_index(
        [("environment", ASCENDING),
         ("architecture", ASCENDING)]
    )

    for env in constants.environments:
        update_sources(repo, env)
        for arch in ['all', 'amd64']:
            update_packages(repo=repo, env=env, arch=arch)


def list_envs(repo):
    fields = {'_id': 0, 'environment': 1}
    envs = set()
    for record in mongo_connection.repos()[repo].find(projection=fields):
        envs.add(record['environment'])
    return list(envs)


def drop(repo):
    conf = Config.find(repo)
    conf.delete()
    mongo_connection.repos()[repo].drop()


def find_recycle_branches(repo, recycle_after):
    q = {
        'environment': re.compile(r'^.*2remove$'),
        'recycle_after': {'$lt': recycle_after}
    }
    a = [
        {'$match': q},
        {'$group': {'_id': '$environment', 'pkg_count': {'$sum': 1}}}
    ]
    return [i['_id'] for i in mongo_connection.repos()[repo].aggregate(a, allowDiskUse=True) if i['pkg_count'] > 0]


def list_architectures(repo, env):
    r = mongo_connection.cacus()[repo].find(
        {'environment': env},
        {'architecture': 1}
    )
    return [x['architecture'] for x in r]
