#!/usr/bin/env python
# -*- coding: utf-8 -*-

import sys
import urllib
from HTMLParser import HTMLParser

import os
import re
import requests
import logging

from infra.dist.cacus.lib import common
from infra.dist.cacus.lib import constants
from infra.dist.cacus.lib import repo_manage
from infra.dist.cacus.lib.dbal import package
from infra.dist.cacus.lib.utils.microdinstall import change_file

log = logging.getLogger(__name__)


class ImportException(Exception):
    pass


def _checkFile(url):
    r = requests.head(url)
    return r.status_code == 200


class RepoDirIndexParser(HTMLParser):
    def __init__(self, base_url):
        HTMLParser.__init__(self)
        self._base_url = base_url
        self._changes_re = re.compile(
            r"(?P<changes>(?P<pkg>[-+.A-Za-z0-9]+)_(?P<ver>[-+~:.A-Za-z0-9]+)_(?P<arch>amd64|all|i386|arm64|powerpc)\.changes)")
        self.changes = []

    # def handle_data(self, data):
    #    m = self._changes_re.search(data)
    #    if m:
    #        self.changes.append(self._base_url + m.group('changes'))
    def handle_starttag(self, tag, attrs):
        if tag == 'a':
            for attr in attrs:
                if attr[0] == 'href':
                    url = urllib.unquote(attr[1])
                    m = self._changes_re.search(url)
                    if m:
                        self.changes.append(
                            self._base_url + m.group('changes'))


def import_package(changefile=None, repo=None, env='unstable'):
    pkg_files = []
    base_url = 'http://dist.yandex.ru/{}/{}'.format(repo, env)
    changes = change_file.ChangeFile()
    changes.load_from_file(changefile)
    filename_list = [x[2] for x in changes.get_files()]
    try:
        p = package.Package.find_one(repo, source=changes['source'], version=changes['version'])
        if p and not common.UploadHelper(p, filename_list).is_extra_upload():
            #           Uncomment next line to import
            #            pass
            #           Comment to import from here
            log.warning("{}_{} is already uploaded to repo '{}', "
                        "environment '{}'".format(changes['source'],
                                                  changes['version'],
                                                  repo,
                                                  p.env)
                        )
            if p.env != env:
                log.warning(
                    "Dmoving {}_{} in repo '{}' from '{}' to '{}'".format(
                        changes['source'],
                        changes['version'],
                        repo,
                        p.env,
                        env
                    )
                )
                repo_manage.dmove_package(
                    package=changes['source'],
                    version=changes['version'],
                    repo=repo,
                    source=p.env,
                    destination=env,
                    skipUpdateMeta=True
                )
            return None
        #       The end
        else:
            log.info(
                "Importing {}_{} to {}/{}".format(
                    changes['source'], changes['version'], repo, env))
    except KeyError as e:
        log.error("Cannot find field {} in {}, skipping package".format(
            e[0], changefile))
        raise ImportException(
            "Cannot find field %s in %s, skipping package", e[0].format(
                changefile)
        )

    for f in filename_list:
        if f.endswith(constants.deb_extensions):
            if f.find('_amd64') >= 0:
                url = '/'.join((base_url, 'amd64', f))
            elif f.find('_all') >= 0:
                url = '/'.join((base_url, 'all', f))
            elif f.find('_i386') >= 0:
                url = '/'.join((base_url, 'i386', f))
            elif f.find('_arm64') >= 0:
                url = '/'.join((base_url, 'arm64', f))
            elif f.find('_powerpc') >= 0:
                url = '/'.join((base_url, 'powerpc', f))
            else:
                log.warning("%s: unknown arch!", f)
                sys.exit(1)
        else:
            url = '/'.join((base_url, 'source', f))

        if not _checkFile(url):
            log.error("%s (%s): file not found", url, f)
            raise ImportException("{} not found".format(url))
        else:
            pkg_files.append(url)

    downloaded = []
    for url in pkg_files:
        file = os.path.join(
            common.config['storage']['temp_dir'], url.split('/')[-1])
        result = common.download_file(url, file)
        if result['result'] != constants.Status.OK:
            [os.unlink(x) for x in downloaded]
            raise ImportException(
                "Cannot download {}: {}".format(url, result['msg']))
        downloaded.append(file)
    downloaded.append(changefile)

    _, err = repo_manage.upload_package(repo, env, downloaded, changes)
    if err:
        log.error("Cannot upload package: %s", err)
        map(os.unlink, downloaded)
        raise ImportException("Cannot upload package: {}".format(err))

    # cleanup
    downloaded.remove(changefile)
    for file in downloaded:
        os.unlink(file)


def import_repo(repo_url=None, repo='common', env='unstable'):
    parser = RepoDirIndexParser(repo_url)
    index = requests.get(repo_url, stream=True, timeout=120)
    for chunk in index.iter_content(64 * 1024):
        parser.feed(chunk)
    changes_files = parser.changes
    log.info("Found %s packages to import", len(changes_files))
    for url in changes_files:
        file = os.path.join(
            common.config['storage']['temp_dir'], url.split('/')[-1])
        result = common.download_file(url, file)
        if result['result'] == constants.Status.OK:
            try:
                import_package(file, repo, env)
            except ImportException as e:
                log.error("Cannot import %s: %s", file, e)
            os.unlink(file)
        else:
            log.error("Cannot download %s", file)

    for arch in ('amd64', 'all', 'i386', 'arm64', 'powerpc'):
        log.info("Updating '%s/%s/%s' repo metadata", repo, env, arch)
        repo_manage.update_repo_metadata(repo=repo, env=env, arch=arch)

    log.info(
        "Import from {} completed, uploaded {} packages to {} {}".format(
            repo_url, len(changes_files), repo, env)
    )
