import datetime
import os

from infra.dist.cacus.lib.notifications import policy
from infra.dist.cacus.lib.notifications import notification


def _fmt_ts(ts):
    return datetime.datetime.fromtimestamp(ts).strftime('%Y-%m-%d %H:%M:%S')


def new_changes(repo, env, package, timestamp, files, uploaded_by):
    subject = '{}: new package {} uploaded to branch {}'.format(repo, package, env)
    processed_files = '\n'.join(map(lambda x: '* {}'.format(os.path.basename(x)), files))
    text = ''' \
Hi!
Package {} has been uploaded to {}/{} by {}
Uploaded at: {}
Uploaded by: {}
processed files:
{}
The package should be available to install in a few minutes.
'''.format(package, repo, env, uploaded_by, _fmt_ts(timestamp), uploaded_by, processed_files)
    return notification.Notification(policy.Type.NEW, repo, subject, text, (uploaded_by,))


def malformed_changes(repo, changes, reason, timestamp, rcpt_list=None):
    subject = '{}: cannon parse {} file!'.format(repo, changes)
    text = '''\
Hi!
Error occured while parsing changes {} for repo: {}.
Uploaded at: {}.
Reason:
{}
'''.format(changes, repo, _fmt_ts(timestamp), reason)
    return notification.Notification(policy.Type.MALFORMED, repo, subject, text, rcpt_list)


def reject_changes(repo, package, timestamp, reason, uploaded_by):
    subject = '{}: package {} rejected'.format(repo, package)
    text = '''\
Hi!
Package {} has been rejected to be uploaded to {}
Uploaded at: {}
Uploaded by: {}
Reason:
{}
'''.format(package, repo, _fmt_ts(timestamp), uploaded_by, reason)
    return notification.Notification(policy.Type.REJECT, repo, subject, text, (uploaded_by,))


def repeated_changes(repo, env, package, timestamp, files, uploaded_by):
    subject = '{}: new files for {} has been uploaded to branch {}'.format(repo, package, env)
    processed_files = '\n'.join(map(lambda x: '* {}'.format(os.path.basename(x)), files))
    text = '''\
Hi!
Some new files for existing package {} has been uploaded to {}/{} by {}
Uploaded at: {}
processed files:
{}
New files from this package should be available to install in a few minutes.
'''.format(package, repo, env, uploaded_by, _fmt_ts(timestamp), processed_files)
    return notification.Notification(policy.Type.REPEAT, repo, subject, text, (uploaded_by,))


def conflict_changes(repo, package, version, current_env, target_env, timestamp, uploaded_by):
    subject = '{}: conflict uploading {}={}'.format(repo, package, version)
    text = '''\
Hi!
Package {} causes conflict uploading to repo {}
Current env: {}
Target env: {}
Uploaded at: {}
Uploaded by: {}
'''.format(package, repo, current_env, target_env, _fmt_ts(timestamp), uploaded_by)
    return notification.Notification(policy.Type.CONFLICT, repo, subject, text, (uploaded_by, ))
