import logging
from infra.dist.cacus.lib import common
from infra.dist.cacus.lib.notifications import policy

log = logging.getLogger(__name__)


class Notification(object):
    def __init__(self, type_, repo, subject, text, rcpt_list=None, policy_fun=policy.default_policy):
        self.policy = policy_fun()
        self.repo = repo
        self.type = type_
        self.rcpts = set()
        rcpt_list = rcpt_list or ()
        self.add_rcpts(rcpt_list)
        self.subject = subject
        self.text = text
        self._sent = False

    def _send(self, rcpt, send_func=common.send_mail):
        log.info('Sending notification to: {}'.format(str(rcpt)))
        send_func(
            common.config['notifications']['mail_from'],
            str(rcpt),
            self.subject,
            self.text
        )

    @staticmethod
    def _extract_addr(addr):
        lb = addr.find('<')
        rb = addr.find('>')
        if lb >= 0 and rb > 0:
            return addr[lb+1:rb]
        else:
            return addr

    def add_rcpt(self, rcpt):
        rcpt = self._extract_addr(rcpt)
        log.debug('Adding rcpt: {}, type: {}, repo: {}'.format(rcpt, self.type, self.repo))
        self.rcpts.add(rcpt)

    def add_rcpts(self, rcpts):
        for r in rcpts:
            self.add_rcpt(r)

    def send(self):
        if not self._sent:
            self.add_rcpts(self.policy.extra_rcpts(self.type, self.repo))
            log.info('Sending notifications...')
            log.debug('Rcpts: {}, type: {}, repo: {}'.format(self.rcpts, self.type, self.repo))
            for rcpt in self.rcpts:
                if self.policy.allowed(self.type, self.repo, rcpt):
                    self._send(rcpt)
                else:
                    log.debug('Skipped notification for {} (repo: {}, type: {})'.format(rcpt, self.repo, self.type))
            self._sent = True
