import logging
from infra.dist.cacus.lib import common

log = logging.getLogger(__name__)


class Type(object):
    NEW = 'upload'  # compatibility with current config
    REJECT = 'reject'
    MALFORMED = 'malformed'
    REPEAT = 'repeat'
    CONFLICT = 'conflict'


class Policy(object):
    def __init__(self):
        self._conf = None

    def configure(self, conf):
        log.debug('Configured default: {}'.format(conf))
        self._conf = conf

    def valid(self):
        return self._conf is not None

    def extra_rcpts(self, type_, repo):
        log.debug('Getting extra rcpts for repo: {}, type: {}'.format(repo, type_))
        if 'repos' in self._conf:
            if repo not in self._conf['repos']:
                log.debug('Extra rcpts not found for repo: {}'.format(repo))
                return tuple()
            if type_ not in self._conf['repos'][repo]:
                log.debug('Extra rcpts not found for repo: {}, type: {}'.format(repo, type_))
                return tuple()
            rcpts = self._conf['repos'][repo][type_]['rcpt']
            log.debug('Extra rcpts found for repo: {}, type: {}: rcpts: {}'.format(repo, type_, rcpts))
            return rcpts
        return tuple()

    def allowed(self, type_, repo, rcpt):
        msg = 'config value'
        if rcpt in self.extra_rcpts(type_, repo):
            allowed = True
            msg = 'rcpt in extra_rcpts'
        elif type_ == Type.NEW:
            allowed = self._conf.get('notify_on_success', False)
        elif type_ == Type.REJECT:
            allowed = self._conf.get('notify_on_reject', True)
        elif type_ == Type.CONFLICT:
            allowed = self._conf.get('notify_on_conflict', True)
        elif type_ == Type.MALFORMED:
            allowed = self._conf.get('notify_on_malformed', True)
        elif type_ == Type.REPEAT:
            allowed = self._conf.get('notify_on_repeat', False)
        else:
            allowed = self._conf.get('send_notifications_by_default', False)
        log.debug('Allowed: {}, type: {}, repo: {}, rcpt: {}, msg: {}'.format(allowed, type_, repo, rcpt, msg))
        return allowed


_policy_instance = Policy()


def default_policy():
    if not _policy_instance.valid():
        _policy_instance.configure(common.config['notifications'])
    return _policy_instance
