#!/usr/bin/env python
# -*- coding: utf-8 -*-

import httplib
import json
import socket
import time
import logging
import base64
import xml.etree.ElementTree as ET
from urlparse import urlparse

# import os
import requests
from os import SEEK_END

from infra.dist.cacus.lib import plugins, common, constants
from infra.dist.cacus.lib.dbal import deleted_key


log = logging.getLogger(__name__)


class MDSStorage(plugins.IStoragePlugin):

    def configure(self, config):
        self.read_url = config['read_url']
        self.write_url = config['write_url']
        self.torent_url = config['torrent_url']
        self.auth_header = config['auth_header']
        self.c_timeout = config['connect_timeout']
        self.r_timeout = config['read_timeout']

    def delete(self, key, deleted_key_store=deleted_key.default_store):
        if key[0] == '/':
            key = key[1:]
        url = "{}/delete-repo/{}".format(self.write_url, key)
        for n_try in xrange(3):
            try:
                response = requests.post(
                    url,
                    headers=self.auth_header,
                    timeout=(self.c_timeout, self.r_timeout)
                )
                if response.status_code not in [200, 404]:
                    response.raise_for_status()
                else:
                    log.warning("Trying to delete inexisting MDS key: %s", url)
                msg = "POST {} {} {}".format(
                    url,
                    response.status_code,
                    response.elapsed.total_seconds()
                )
                log.info(msg)
                return None
            except (requests.exceptions.ConnectionError,
                    requests.exceptions.HTTPError,
                    requests.exceptions.Timeout,
                    IOError) as e:
                isConnectionErr = isinstance(
                    e,
                    requests.exceptions.ConnectionError
                )
                isHTTPErr = isinstance(e, requests.exceptions.HTTPError)
                isIOErr = isinstance(e, IOError)
                isTimeoutErr = isinstance(e, requests.exceptions.Timeout)
                if isConnectionErr or isHTTPErr or isIOErr:
                    log.error("Error deleting %s: %s", url, e)
                if isTimeoutErr:
                    log.error("Timeout deleting %s: %s", url, e)
            time.sleep(3 ** n_try)
        else:
            deleted_key_store.push(key)
            return 'failed to delete key: {}'.format(key)

    def put(self, key, filename=None, file=None, size=None, md5=None):

        url_obj = urlparse(self.write_url)
        conn = httplib.HTTPConnection(
            url_obj.netloc,
            timeout=self.r_timeout
        )
        url = "{0}/upload-repo/{1}".format(self.write_url, key)
        storage_key = None
        resp = None
        resp_body = None

        if filename:
            file = open(filename, 'rb')

        with file as f:
            path = "{0}/upload-repo/{1}".format(url_obj.path, key)
            f.seek(0, SEEK_END)
            data_len = f.tell()
            if size is not None:
                if data_len != int(size):
                    raise IOError('file: {} size: {} expected: {}'.format(filename, data_len, size))
            for n_try in xrange(3):
                f.seek(0)
                try:
                    conn.connect()
                    conn.putrequest('POST', path)
                    headers = {
                        'Authorization': self.auth_header['Authorization'],
                        'User-Agent': 'сacus',
                        'Accept': '*/*',
                        'Content-Length': str(data_len)
                    }
                    if md5 is not None:
                        log.debug('md5 is not empty, will add Content-MD5 header to request', md5)
                        md5_64 = base64.b64encode(md5.decode('hex'))
                        log.debug('got base64 version of md5: %s : %s', md5, md5_64)
                        headers['Contend-MD5'] = md5_64.decode('utf-8')
                    for hdr_name, hdr_val in headers.iteritems():
                        conn.putheader(hdr_name, hdr_val)
                    conn.endheaders()
                    while True:
                        piece = f.read(4096)
                        if not piece:
                            break
                        conn.send(piece)
                    resp = conn.getresponse()
                    resp_body = resp.read()
                    conn.close()

                except socket.timeout:
                    log.error("Timeout uploading %s", url)
                    continue
                except httplib.HTTPException as e:
                    log.error("Error uploading %s: %s", url, e)
                    continue
                except IOError as e:
                    if not resp:
                        try:
                            resp = conn.getresponse()
                            conn.close()
                            log.error(
                                "Error uploading {}: {}, "
                                "HTTP status: {}".format(url, e, resp.status)
                            )
                            if resp.status != 403:
                                continue
                        except (
                                IOError, httplib.HTTPException, socket.timeout
                        ) as e:
                            msg = "Error handling non-standard situation," \
                                  " during upload {}: {}".format(url, e)
                            log.error(msg)
                            continue
                    else:
                        continue

                log.info("POST %s %s", url, resp.status)
                if resp.status not in [403, 200]:
                    log.error(
                        "Error uploading {}: {}".format(
                            url,
                            resp.status
                        )
                    )
                    continue

                if resp.status == 403:
                    log.warning(
                        "Key {} already exists in MDS, getting it".format(url)
                    )
                    key = None
                    try:
                        xml_response = ET.fromstring(resp_body)
                        key = xml_response.find('key')
                    except (TypeError, SyntaxError, AttributeError):
                        log.error(
                            "Error uploading {}: malformed XML"
                            " response from server".format(url)
                        )
                        continue
                    if key is not None:
                        storage_key = key.text
                        log.debug("Got storage key '%s'", storage_key)
                        return storage_key
                    else:
                        log.error(
                            "No key field found in response! "
                            "Response was: {}".format(resp_body)
                        )
                        return None
                    continue
                if resp.status == 200:
                    xml_response = None
                    try:
                        xml_response = ET.fromstring(resp_body)
                    except (SyntaxError, TypeError):
                        log.error(
                            "Error uploading {}: malformed XML"
                            " response from server".format(url)
                        )
                        continue
                    break

                time.sleep(3 ** n_try)
            else:
                log.critical("Cannot upload %s", filename)
                return None

            try:
                storage_key = xml_response.attrib['key']
            except (KeyError, AttributeError):
                log.error(
                    "Error uploading {}: malformed XML"
                    " response from server".format(url)
                )
                return None
            log.debug("Got storage key '%s'", storage_key)

        return storage_key

    def generate_torrent(self, key, filename):
        url = self.torent_url
        post_data = [{'ns': 'repo',
                      'mds_key': key,
                      'type': 'file',
                      'path': filename}]
        for n_try in xrange(3):
            try:
                response = requests.post(
                    url,
                    timeout=(self.c_timeout, self.r_timeout),
                    data=json.dumps(post_data)
                )
                response.raise_for_status()
                msg = "POST {} {} {} {}".format(
                    url,
                    filename,
                    response.status_code,
                    response.elapsed.total_seconds()
                )
                log.info(msg)
                return response.content
            except (requests.exceptions.ConnectionError,
                    requests.exceptions.HTTPError,
                    requests.exceptions.Timeout,
                    IOError) as e:
                isConnectionErr = isinstance(
                    e,
                    requests.exceptions.ConnectionError
                )
                isHTTPErr = isinstance(e, requests.exceptions.HTTPError)
                isIOErr = isinstance(e, IOError)
                isTimeoutErr = isinstance(e, requests.exceptions.Timeout)
                if isConnectionErr or isHTTPErr or isIOErr:
                    log.error("Error generating torrent %s: %s", url, e)
                if isTimeoutErr:
                    log.error("Timeout generating torrent %s: %s", url, e)
            time.sleep(3 ** n_try)

    # WTF is this?
    # def get(self, key):
    #     return os.path.join(self.root, key)

    def get(self, key):
        url = "{}/get-repo/{}".format(self.read_url, key)
        for n_try in xrange(3):
            try:
                response = requests.get(
                    url,
                    headers=self.auth_header,
                    timeout=(self.c_timeout, self.r_timeout))
                log.info(
                    "GET %s %s %s",
                    url,
                    response.status_code,
                    response.elapsed.total_seconds())
                response.raise_for_status()
                return response
            except requests.exceptions.ConnectionError as e:
                log.error("Error requesting %s: %s", url, e)
            except requests.exceptions.HTTPError as e:
                log.error("Error requesting %s: %s", url, e)
            except requests.exceptions.Timeout as e:
                log.error("Timeout requesting %s: %s", url, e)
            except IOError as e:
                log.error("Error requesting %s: %s", url, e)

            time.sleep(3 ** n_try)

    def copy(self, src, dst):
        response = self.get(src)
        content_file = common.myStringIO()
        content_file.write(response.content)
        return self.put(dst, file=content_file)

    def rename(self, old_key, new_key):
        response = self.get(old_key)
        content_file = common.myStringIO()
        content_file.write(response.content)
        storage_key = self.put(new_key, file=content_file)
        self.delete(old_key)
        return storage_key

    def cleanify_url(self, dirty_key):
        return dirty_key.replace('/storage', '', 1)

    # in this section resides code,
    # that returns to user appropriate fiel from MDS

    def give_storage_file(self, storage_key, request):
        '''Writes redirect to special nginx location for given request'''
        if storage_key == constants.MDS_EMPTY_KEY:
            return
        url = "{0}{1}".format('/storage/', storage_key)
        log.debug("Redirecting to %s", url)
        request.add_header("X-Accel-Redirect", url)

    def complete_url_regex(self, url_re):
        return r"/storage/(?P<couple_id>\d+)/{}".format(url_re)
