#!/usr/bin/env python
# -*- coding: utf-8 -*-


import email.utils
import inspect
import logging
import time
from json import dumps

import os
import motor
from concurrent.futures import ThreadPoolExecutor
from tabulate import tabulate
import tornado
import tornado.netutil
import tornado.process
from tornado import gen, httputil, httpserver
from tornado.ioloop import IOLoop
from tornado.web import (
    RequestHandler, Application, url, asynchronous, MissingArgumentError)
from infra.dist.cacus.lib import common
from infra.dist.cacus.lib import constants
from infra.dist.cacus.lib import loader
from infra.dist.cacus.lib import repo_manage
from infra.dist.cacus.lib import dist_importer
from infra.dist.cacus.lib.dbal import mongo_connection
from infra.dist.cacus.lib.dbal import package_repository
from infra.dist.cacus.lib.dbal import errors


access_log = logging.getLogger('infra.dist.cacus.lib.common.repo-daemon.access')
app_log = logging.getLogger('infra.dist.cacus.lib.common.repo-daemon.application')
gen_log = logging.getLogger('infra.dist.cacus.lib.common.repo-daemon.general')


pkg_search_projection = {
    '_id': 0,
    'Source': 1,
    'environment': 1,
    'Version': 1,
    'debs.maintainer': 1,
    'debs.Architecture': 1,
    'debs.Package': 1,
    'debs.Description': 1
}
arlg = None
areg = None


def dist_push(repo=None, changes=None):
    gen_log.info("Got push for repo %s file %s", repo, changes)
    try:
        base_dir = package_repository.Config.find(repo).incoming_dir
    except errors.RepositoryNotFound:
        gen_log.error("Cannot find repo %s", repo)
        return {'result': constants.Status.NOT_FOUND, 'msg': "No such repo"}

    filename = os.path.join(base_dir, changes.split('/')[-1])
    url = "http://dist.yandex.ru/{}/unstable/{}".format(repo, changes)
    result = common.download_file(url, filename)
    if result['result'] == constants.Status.OK:
        try:
            dist_importer.import_package(filename, repo, 'unstable')
        except dist_importer.ImportException as e:
            return {'result': constants.Status.ERROR, 'msg': str(e)}
        return {'result': constants.Status.OK, 'msg': 'Imported successfully'}
    else:
        return result


class CachedRequestHandler(RequestHandler):

    # I don't know what is this:
    # def prepare(self):
    #     pass

    @gen.coroutine
    def _cache_expired(self, repo, env, arch='__all__'):
        db = self.settings['db_cacus']
        latest_dt = None
        selector = {'environment': env}
        if (arch != '__all__'):
            selector['architecture'] = arch
        repos = db[repo].find(selector, {'lastupdated': 1})
        while (yield repos.fetch_next):
            dt = repos.next_object()['lastupdated']
            if not latest_dt or dt > latest_dt:
                latest_dt = dt

        if_modified = self.request.headers.get('If-Modified-Since')
        if not if_modified or not latest_dt:
            raise gen.Return((True, latest_dt))

        cached_dt = email.utils.parsedate(if_modified)
        cached_ts = time.mktime(cached_dt)
        latest_ts = time.mktime(latest_dt.timetuple())
        if latest_ts <= cached_ts:
            raise gen.Return((False, latest_dt))
        else:
            raise gen.Return((True, latest_dt))


class PackagesHandler(CachedRequestHandler):
    @gen.coroutine
    def get(self, repo=None, env=None, arch=None, ext=None):
        db = self.settings['db_cacus']
        (expired, dt) = yield self._cache_expired(repo, env, arch)
        if not expired:
            self.set_status(304)
            return
        if dt:
            self.add_header("Last-Modified", httputil.format_timestamp(dt))

        doc = yield db[repo].find_one(
            {'environment': env, 'architecture': arch})

        if doc:
            storage_plugin = loader.get_plugin('storage')
            app_log.info("Will redirect from %s/%s/%s/Packages", repo,
                         env, arch)

            if ext == ".gz":
                storage_plugin.give_storage_file(doc['gziped_packages'], self)
            elif ext == ".bz2":
                storage_plugin.give_storage_file(doc['bziped_packages'], self)
            else:
                storage_plugin.give_storage_file(doc['packages_file'], self)

            self.set_status(200)
        else:
            if repo in (yield db.list_collection_names()):
                self.set_status(200)
            else:
                self.set_status(404)


class SourcesHandler(CachedRequestHandler):
    '''Handler for giving Sources file'''
    @gen.coroutine
    def get(self, repo=None, env=None, ext=None):
        db = self.settings['db_cacus']
        (expired, dt) = yield self._cache_expired(repo, env, 'source')
        if not expired:
            self.set_status(304)
            return
        if dt:
            self.add_header("Last-Modified", httputil.format_timestamp(dt))

        doc = yield db[repo].find_one(
            {'environment': env, 'architecture': 'source'})

        if doc:
            storage_plugin = loader.get_plugin('storage')
            app_log.info("Will redirect from %s/%s/source/Sources", repo,
                         env)

            if ext == ".gz":
                storage_plugin.give_storage_file(doc['gziped_sources'], self)
            elif ext == ".bz2":
                storage_plugin.give_storage_file(doc['bziped_sources'], self)
            else:
                storage_plugin.give_storage_file(doc['sources_file'], self)

            self.set_status(200)
        else:
            if repo in (yield db.list_collection_names()):
                self.set_status(200)
            else:
                self.set_status(404)


class SourcesFilesHandler(CachedRequestHandler):
    '''Handler for giving files, listed in Sources file'''

    @gen.coroutine
    def get(self, repo=None, env=None, file=None):
        db = self.settings['db_repos']
        doc = yield db[repo].find_one(
            {'environment': env, 'sources.name': file},
            {'sources.storage_key': 1, 'sources.name': 1}
        )
        if not doc or not ('sources' in doc):
            self.set_status(404)
            return
        for f in doc['sources']:
            if f['name'] == file:
                # storage_plugin = loader.get_plugin('storage')
                app_log.info("Redirecting %s", file)
                self.add_header("X-Accel-Redirect", f['storage_key'])
                # storage_plugin.give_storage_file(f['storage_key'], self)
                break
        self.set_status(200)


class TorrentHandler(RequestHandler):
    @gen.coroutine
    def get(self, **kwargs):
        repo = kwargs['repo'] if 'repo' in kwargs else None
        if not repo:
            self.set_status(404)
            return
        rbtorrent_falg = self.get_argument('rbtorrent')
        if rbtorrent_falg != '1':
            self.set_status(404)
            return
        db = self.settings['db_repos']
        storage_key = self.request.uri
        storage_key = storage_key[:len(storage_key) - len('?rbtorrent=1')]
        doc = yield db[repo].find_one(
            {'debs.storage_key': storage_key},
            {'debs.rbtorrent_id': True, 'debs.storage_key': True}
        )
        if not doc or not ('debs' in doc):
            self.set_status(404)
            return
        for deb in doc['debs']:
            if deb['storage_key'] == storage_key:
                self.add_header("X-Resource-Id", deb['rbtorrent_id'])
                self.set_status(204)
                return
        self.set_status(404)


class ReleaseHandler(CachedRequestHandler):
    @asynchronous
    @gen.coroutine
    def get(self, repo=None, env=None, arch=None, release_file=None):
        db = self.settings['db_cacus']
        (expired, dt) = yield self._cache_expired(repo, env, arch)
        if not expired:
            self.set_status(304)
            return
        if dt:
            self.add_header("Last-Modified", httputil.format_timestamp(dt))

        doc = yield db[repo].find_one(
            {'environment': env, 'architecture': arch})
        if doc:
            if release_file == 'InRelease':
                if 'in_release_file' in doc:
                    self.write(doc['in_release_file'])
                else:
                    # trick for lua-cache
                    self.set_status(200)
            elif release_file == 'Release':
                self.write(doc['release_file'])
            elif release_file == 'Release.gpg':
                self.write(doc['release_gpg'])
            else:
                self.set_status(404, 'not implemented')


class ApiMoveHandler(RequestHandler):
    def _convert_to_bool(slef, arg_name, val):
        if val.lower() == 'true':
            return True
        elif val.lower() == 'false':
            return False
        else:
            raise MissingArgumentError(arg_name)

    def parse_args(self, kwarg_list, internal_args):
        fields_to_type_conversions = {
            'delete': self._convert_to_bool
        }
        kwargs = {}
        # yay, i'm too lazy to use orderedsets
        external_args = [
            kwarg if kwarg not in internal_args else None
            for kwarg in kwarg_list]
        external_args = filter(lambda x: bool(x), external_args)
        for kwarg in external_args:
            kwargs[kwarg] = self.get_argument(kwarg, default=None)
            if kwarg in fields_to_type_conversions:
                type_conversion = fields_to_type_conversions[kwarg]
                try:
                    kwargs[kwarg] = type_conversion(kwarg, kwargs[kwarg])
                except MissingArgumentError:
                    kwargs[kwarg] = None
            if kwargs[kwarg] is None:
                self.set_status(400)
                self.write(
                    {
                        'success': False,
                        'msg': 'arguments {} are mandatory'.format(
                            external_args)
                    }
                )
                return False
        return kwargs

    @asynchronous
    @gen.coroutine
    def post(self, key=None):
        internal_args = ['skipUpdateMeta']
        if key == 'd':
            func = repo_manage.dmove_package
        if key == 'r':
            func = repo_manage.rmove_package
        if key == 'b':
            func = repo_manage.bmove_package

        kwargs = self.parse_args(inspect.getargspec(func).args, internal_args)
        if not kwargs:
            raise gen.Return()

        r = yield self.settings['workers'].submit(func, **kwargs)

        if r['result'] == constants.Status.OK:
            self.write({'success': True, 'msg': r['msg']})
        elif r['result'] == constants.Status.NO_CHANGES:
            self.write({'success': True, 'msg': r['msg']})
        elif r['result'] == constants.Status.NOT_FOUND:
            self.set_status(404)
            self.write({'success': False, 'msg': r['msg']})
        elif r['result'] == constants.Status.TIMEOUT:
            self.set_status(408)
            self.write({'success': False, 'msg': r['msg']})


class ApiDistPushHandler(RequestHandler):

    @asynchronous
    @gen.coroutine
    def post(self, repo=None):
        changes_file = self.get_argument('file')
        db = self.settings['db_cacus']
        if repo in (yield db.list_collection_names()):
            self.write({
                'success': True,
                'msg': 'Submitted package import job'
            })
        else:
            self.set_status(404)
            self.write({
                'success': False,
                'msg': "Repo {} is not configured".format(repo)
            })

        r = yield self.settings['workers'].submit(
            dist_push, repo=repo, changes=changes_file
        )
        if r['result'] == constants.Status.OK:
            self.write({'success': True, 'msg': r['msg']})
        elif r['result'] == constants.Status.NOT_FOUND:
            self.set_status(404)
            self.write({'success': False, 'msg': r['msg']})
        else:
            self.set_status(500)
            self.write({'success': False, 'msg': r['msg']})


class SearchParameters(object):
    def __init__(self, repo, pkg, ver, env, descr,
                 lang, strict, sources, withurls, human):
        self.repo = repo
        self.pkg = pkg
        self.ver = ver
        self.env = env
        self.descr = descr
        self.lang = lang
        self.strict = strict
        self.sources = sources
        self.withurls = withurls
        self.human = human


class ApiSearchHandler(RequestHandler):
    @asynchronous
    @gen.coroutine
    def get(self):
        search_parameters = self.parse_args()
        found = yield self.search(search_parameters)
        result = {}
        if not found:
            self.set_status(404)
            result = {'success': False, 'result': []}
        else:
            result = self.format(search_parameters, found)
        self.write(result)

    def generate_search_projection(self, sp):
        search_projection = pkg_search_projection.copy()

        if sp.human:
            search_projection = {
                '_id': 0, 'debs.storage_key': 1, 'environment': 1
            }
        if sp.withurls:
            search_projection['debs.storage_key'] = 1
            search_projection['sources.storage_key'] = 1
        if sp.sources:
            search_projection['sources.name'] = 1
        return search_projection

    def generate_selector(self, sp):
        selector = {}
        if sp.pkg:
            selector['$or'] = []

        if not sp.strict:
            if sp.pkg:
                selector['$or'].append({'Source': {'$regex': sp.pkg}})
                selector['$or'].append({'debs.Package': {'$regex': sp.pkg}})
            if sp.sources and sp.pkg:
                selector['$or'].append({'sources.name': {'$regex': sp.pkg}})
        else:
            if sp.pkg:
                selector['$or'].append({'Source': sp.pkg})
                selector['$or'].append({'debs.Package': sp.pkg})
            if sp.sources and sp.pkg:
                selector['$or'].append({'sources.name': sp.pkg})

        if sp.ver:
            selector['Version'] = (
                {'$regex': sp.ver} if not sp.strict else sp.ver)
        if sp.env:
            selector['environment'] = sp.env
        if sp.descr:
            if sp.lang:
                selector['$text'] = {'$search': sp.descr, '$language': sp.lang}
            else:
                selector['$text'] = {'$search': sp.descr}
        return selector

    @gen.coroutine
    def search(self, sp):
        app_log.debug('got search request with params: {}'.format(vars(sp)))
        db = self.settings['db_repos']
        app_log.debug('api.search db type: {}'.format(type(db)))
        # repos = (yield db.collection_names(
        #     include_system_collections=False)) if not sp.repo else [sp.repo]
        repos = (yield db.list_collection_names()) if not sp.repo else [sp.repo]
        selector = self.generate_selector(sp)
        app_log.debug('prepared db selector: {}'.format(selector))
        search_projection = self.generate_search_projection(sp)
        app_log.debug('prepared db projection: {}'.format(search_projection))

        pkgs = []

        for repository in repos:
            app_log.debug('fetching results for repo: {}'.format(repository))
            cursor = db[repository].find(selector, search_projection)
            while (yield cursor.fetch_next):
                pkg = cursor.next_object()
                if pkg:
                    app_log.debug('got pkg: {}'.format(pkg))
                    p = dict((k.lower(), v) for k, v in pkg.iteritems())
                    p['debs'] = [
                        dict(
                            (k.lower(), v) for k, v in deb.iteritems()
                        ) for deb in pkg['debs']
                    ]
                    p['repository'] = repository
                    pkgs.append(p)

        raise gen.Return(pkgs)

    # When withurls is turned on, human=false does not works
    def format(self, sp, found):
        if sp.withurls:
            found = self.format_urls(found)
        if sp.human:
            results = [['repo', 'env', 'package']]
            if sp.withurls:
                results[0].append('url')
            for pkg in found:
                for deb in pkg['debs']:
                    res_record = [
                        pkg['repository'],
                        pkg['environment'],
                        os.path.basename(deb['storage_key']),
                    ]
                    if sp.withurls:
                        res_record.append(deb['storage_key'])
                    results.append(res_record)
                if sp.sources and 'sources' in pkg:
                    for src in pkg['sources']:
                        res_record = [
                            pkg['repository'],
                            pkg['environment'],
                            src['name']
                        ]
                        if sp.withurls:
                            res_record.append(src['storage_key'])
                        results.append(res_record)

            result = tabulate(results, headers="firstrow", tablefmt="psql")
            result += '\n'
        else:
            result = {'success': True, 'result': found}
        return result

    def format_urls(self, pkgs):
        fqdn = common.config['repo_daemon']['cacus_fqdn']
        for pkg_idx, pkg_rec in enumerate(pkgs):
            for deb_idx, deb_rec in enumerate(pkg_rec['debs']):
                pkgs[pkg_idx]['debs'][deb_idx]['storage_key'] = \
                    fqdn + deb_rec['storage_key']
            if 'sources' in pkg_rec:
                for src_idx, src_rec in enumerate(pkg_rec['sources']):
                    pkgs[pkg_idx]['sources'][src_idx]['storage_key'] = \
                        fqdn + src_rec['storage_key']
        return pkgs

    def parse_args(self):
        boolify = (lambda b: True if b.lower() == 'true' else False)
        repo = self.get_argument('repo', '')
        pkg = self.get_argument('pkg', '')
        ver = self.get_argument('ver', '')
        env = self.get_argument('env', '')
        descr = self.get_argument('descr', '')
        lang = self.get_argument('lang', '')
        strict = boolify(self.get_argument('strict', '').lower())
        sources = boolify(self.get_argument('sources', '').lower())
        withurls = boolify(self.get_argument('withurls', '').lower())
        human = boolify(self.get_argument('human', '').lower())

        sp = SearchParameters(repo, pkg, ver, env, descr, lang,
                              strict, sources, withurls, human)
        return (sp)


class ApiConductorHandler(RequestHandler):

    @gen.coroutine
    def find_pkg(self, repo, query):
        db = self.settings['db_repos']
        pkg = yield db[repo].find_one(query, pkg_search_projection)
        if not pkg:
            raise gen.Return(None)
        keys_to_lower_case = (
            lambda d: dict((k.lower(), v) for k, v in d.iteritems()))
        p = keys_to_lower_case(pkg)
        p['debs'] = [
            keys_to_lower_case(deb)
            for deb in pkg['debs']
        ]
        raise gen.Return(p)

    @asynchronous
    @gen.coroutine
    def get(self, repo=None):
        pkg = self.get_argument('pkg', '')
        ver = self.get_argument('ver', '')

        if (not pkg) or (not ver):
            err_msg = 'Cannot handle conductor request without pkg or ver'
            app_log.info(err_msg)
            result = {'success': False, 'result': err_msg}
            return

        result = {}
        msg = 'Handling conductor search request for repo: {} package: {}' \
              ' version: {}'.format(repo, pkg, ver)
        app_log.debug(msg)

        query = {'$and': [
            {'Source': pkg},
            {'Version': ver}
        ]}
        package = yield self.find_pkg(repo, query)
        if not package:
            query = {'$and': [
                {'debs.Package': pkg},
                {'Version': ver}
            ]}
            package = yield self.find_pkg(repo, query)

        if not package:
            result = {'success': False, 'result': []}
        else:
            result = {'success': True, 'result': package}
        self.write(result)


class ApiReposJson(RequestHandler):
    @asynchronous
    @gen.coroutine
    def get(self):
        repos_json = []
        for repo in (yield arlg.get_repos()):
            repo_dict = {'branches': [],
                         'type': 'debian',
                         'name': repo,
                         'dmove': repo}
            for env in (yield areg.get_envs(repo)):
                repo_dict['branches'].append(env)
            repos_json.append(repo_dict)
        self.write(dumps(repos_json))


class PaymentRequired(RequestHandler):
    @asynchronous
    @gen.coroutine
    def get(self):
        self.set_status(402)


class TranslationsHandler(RequestHandler):
    @asynchronous
    @gen.coroutine
    def get(self, repo=None, env=None, arch=None, lang=None, compression=None):
        self.set_status(200)


class PingHandler(RequestHandler):

    @asynchronous
    @gen.coroutine
    def get(self):
        self.set_status(200)


def make_app():
    # base = common.config['repo_daemon']['repo_base']
    storage_plugin = loader.get_plugin('storage')
    repo_re = r"(?P<repo>[-_.\w]+)"
    env_re = r"(?P<env>[-_\w]+)"
    arch_re = r"(?P<arch>\w+)"
    deb_name_re = r"(?P<deb_name>.+)"

    packages_re = r"/{}/{}/{}/Packages(?P<ext>\.gz|\.bz2)?$".format(
        repo_re, env_re, arch_re)
    release_re = r"/{}/{}/{}/(?P<release_file>InRelease|Release|Release.gpg)$".format(
        repo_re, env_re, arch_re)
    torrent_re = r"{}/{}$".format(repo_re, deb_name_re)
    torrent_re = storage_plugin.complete_url_regex(torrent_re)
    not_implemented_re = r"/[-.\w]+/\w+/source/Sources\.(?:xz|lz4|lzma|diff/Index)$"
    translations_re = r"/{}/{}/{}/(?P<lang>.*)\.?(?P<compression>gz|bz2|lzma|xz)?$".format(repo_re, env_re, arch_re)
    sources_re = r"/{}/{}/source/Sources(?P<ext>\.gz|\.bz2)?$".format(
        repo_re, env_re)
    sources_files_re = r"/{}/{}/source/(?P<file>.*)$".format(repo_re, env_re)

    api_move_re = r"/api/v1/(?P<key>[bdr])move$"
    # api_dmove_re = r"/api/v1/dmove$"
    api_search_re = r"/api/v1/search$"
    api_conductor_search_re = r"/api/v1/conductor-search/{}$".format(repo_re)
    api_dist_push_re = r"/api/v1/dist-push/{}$".format(repo_re)
    api_repos_json_re = r"/api/v1/repos\.json$"

    ping_re = '/ping'

    return Application([
        url(packages_re, PackagesHandler),
        url(release_re, ReleaseHandler),
        url(torrent_re, TorrentHandler),
        url(not_implemented_re, PaymentRequired),
        url(sources_re, SourcesHandler),
        url(sources_files_re, SourcesFilesHandler),
        # url(api_dmove_re, ApiDmoveHandler),
        url(api_move_re, ApiMoveHandler),
        url(api_search_re, ApiSearchHandler),
        url(api_conductor_search_re, ApiConductorHandler),
        url(api_dist_push_re, ApiDistPushHandler),
        url(api_repos_json_re, ApiReposJson),
        url(ping_re, PingHandler),
        url(translations_re, TranslationsHandler)
    ])


def start_daemon():
    global arlg
    global areg
    app = make_app()
    sockets = tornado.netutil.bind_sockets(common.config['repo_daemon']['port'])
    children = []
    master_process = True
    for i in range(0, common.config['repo_daemon']['worker_processes']):
        pid = os.fork()
        if pid == 0:
            master_process = False
            break
        children.append(pid)
    if master_process:
        while True:
            respawn_workers = []
            app_log.debug('master: checking children for liveness')
            for child in children:
                app_log.debug('master: checking child: {}'.format(child))
                pid, status = os.waitpid(child, os.WNOHANG)
                if pid != 0 and (os.WIFSIGNALED(status) or os.WIFEXITED(status)):
                    app_log.error('worker with pid {} is dead, respawning'.format(child))
                    respawn_workers.append(child)
                    continue
                app_log.debug('master: child {} seems to be alive'.format(child))
            for dead in respawn_workers:
                children.remove(dead)
                app_log.info('respawning old worker {}'.format(dead))
                pid = os.fork()
                if pid == 0:
                    master_process = False
                    break
                app_log.info('respawned worker with pid: {}'.format(pid))
                children.append(pid)
            if not master_process:
                break
            time.sleep(5)
        if master_process:
            exit(0)

    server = httpserver.HTTPServer(app)
    server.add_sockets(sockets)
    mongo_connection.configure(common.config['metadb'], driver=motor.MotorClient, wrapper=None)
    arlg = common.AsyncRepoListGetter(mongo_connection.repos())
    areg = common.AsyncRepoEnvsGetter(mongo_connection.repos())
    thread_pool = ThreadPoolExecutor(common.config['repo_daemon']['worker_threads'])
    app.settings['db_cacus'] = mongo_connection.cacus()
    app.settings['db_repos'] = mongo_connection.repos()
    app.settings['workers'] = thread_pool
    try:
        IOLoop.current().start()
    except KeyboardInterrupt:
        app_log.info('requested keyboard interrupt... exiting...')
        IOLoop.current().stop()
        exit(13)
