#!/usr/bin/env python
# -*- coding: utf-8 -*-

import bz2
import contextlib
import gzip
import hashlib
import json
import logging
import multiprocessing
import os
import random
import re
import stat
import threading
import time
from binascii import hexlify
from datetime import datetime
from functools import partial
from multiprocessing import Process, Lock, Queue

import kazoo.exceptions
from bson import binary
from concurrent.futures import ThreadPoolExecutor
from debian import debfile, deb822
from setproctitle import setproctitle

from infra.dist.cacus.lib import common
from infra.dist.cacus.lib import constants
from infra.dist.cacus.lib import loader
from infra.dist.cacus.lib.dbal import mongo_connection
from infra.dist.cacus.lib.dbal import package
from infra.dist.cacus.lib.dbal import package_repository
from infra.dist.cacus.lib.dbal import package_index
from infra.dist.cacus.lib.dbal import deleted_key
from infra.dist.cacus.lib.notifications import factory as notifications
from infra.dist.cacus.lib.stats.accumulator import Accumulator
from infra.dist.cacus.lib.stats.histogram import Histogram
from infra.dist.cacus.lib.stats.synchronized_accumulator import SynchronizedAccumulator
from infra.dist.cacus.lib.stats.synchronized_histogram import SynchronizedHistogram
from infra.dist.cacus.lib.stats.timing_decorator import callable_timing_hram
from infra.dist.cacus.lib.utils import audit
from infra.dist.cacus.lib.utils.push_metrics import push_metric
from infra.dist.cacus.lib.utils import process

log = logging.getLogger(__name__)

sources_daemon_indexed_repo_count = SynchronizedAccumulator('sources_daemon_indexed_repo_count_summ')
sources_daemon_indexing_time = SynchronizedHistogram('sources_daemon_indexing_time_ahhh')
sources_daemon_indexing_time_prev = [
    sources_daemon_indexing_time.name,
    [0, 0]
]


def dump_sources_daemon_stats(file_path):
    global sources_daemon_indexing_time_prev
    results = [[
        sources_daemon_indexed_repo_count.name,
        sources_daemon_indexed_repo_count.get_value()
    ]]
    if sources_daemon_indexing_time.has_values():
        sources_daemon_indexing_time_prev = [
            sources_daemon_indexing_time.name,
            sorted(sources_daemon_indexing_time.read_values().items(), key=lambda x: x[0])
        ]
    results.append(sources_daemon_indexing_time_prev)
    with open(file_path, 'w') as f:
        json.dump(results, f)


class UploadPackageError(Exception):
    def __init__(self, *args, **kwargs):
        self.msg = args[0]
        if 'retryable' in kwargs:
            self.retryable = kwargs['retryable']
        else:
            self.retryable = False


class UpdateRepoMetadataError(Exception):
    pass


# there is no guarantee, that package will be uploaded to prefered envirinment
def upload_package(repo, prefered_env, files, changes, skipUpdateMeta=False, audit_meta=None, checksums=None,
                   index_store=package_index.default_store):
    audit_meta = audit_meta or []

    def upload_deb(file, size=None, md5=None):
        filename = os.path.basename(file)
        base_key = "{0}/{1}".format(repo, filename)
        retry_count = 10

        log.info(
            "Uploading %s to repo '%s' environment '%s'", base_key, repo, env)

        for n_try in xrange(1, retry_count + 1):
            if size is not None and md5 is not None:
                storage_key = loader.get_plugin('storage').put(
                    base_key, filename=file, size=size, md5=md5
                )
            else:
                storage_key = loader.get_plugin('storage').put(
                    base_key, filename=file
                )
            if storage_key:
                return storage_key, None
            msg = "Error uploading {}, will retry {} times".format(
                file, retry_count - n_try
            )
            log.critical(msg)
        else:
            return None, UploadPackageError(
                "Cannot upload {0} to storage. May be retry will "
                "be performed for whole package".format(file),
                retryable=True
            )

    def do_generate_torrent(key, filename):
        log.info("Geterating rbtorrent for: {}".format(key))
        retry_count = 10
        for n_try in xrange(1, retry_count):
            storage_key = loader.get_plugin('storage').generate_torrent(
                key, filename)
            if storage_key:
                return storage_key, None
            msg = "Error generating rbtorrent {}, will retry {} times".format(
                file, retry_count - n_try
            )
            log.critical(msg)
        else:
            return None, UploadPackageError(
                "Cannot generate rbtorrent for storage key {}. May be "
                "retry will be performed for whole package".format(key),
                retryable=True
            )
    env = prefered_env
    repo_conf = package_repository.Config.find(repo)

    # 1 decide, if it is "extra debs" or "re-upload"
    # 2 if current uplad is "extra deb" upload, we must check if all packages
    # without archtecture specification are already present in database
    # 3 if we already have the same packages without arch specification,
    # upload proper new packages
    # 4 update metadata in environment, that package belongs
    upload_user = changes['changed-by']
    for i in audit_meta:
        if isinstance(i, dict):
            if 'user' in i.keys():
                upload_user = i['user']
    p = package.Package.find_one(repo, source=changes['source'], version=changes['version'])
    if p and p.env != env:
        return notifications.conflict_changes(
            repo,
            p.source,
            p.version,
            p.env,
            env,
            time.time(),
            changes['changed-by']
        ), UploadPackageError('conflict uploading package {}={} to {} branch {} (current env is {})'.format(
            p.source, p.version, repo, env, p.env), retryable=False)
    if p:
        # if flow control is here, we have extra debs or re-upload
        notification = notifications.repeated_changes(
            repo, env, changes['source'], time.time(), files, changes['changed-by']
        )
        upload_helper = common.UploadHelper(p, files)
        # more details in HOSTMAN-268
        if repo_conf.loose_upload_checks:
            extra_upload = lambda: True
        else:
            extra_upload = upload_helper.is_extra_upload
        if extra_upload():
            msg = 'Package {}_{} is already in repository {} environment {}, ' \
                  'so new files will be just added to existing package'.format(
                changes['source'],
                changes['version'],
                repo,
                p.env,
            )
            log.warning(msg)
            env = p.env
            files = upload_helper.generate_files_diff()
            overridden_files = upload_helper.overridden_files()
            if overridden_files:
                log.warn('overriding files {} in package {}={} uploaded_by {}'.format(
                    ','.join(overridden_files), p.source, p.version, upload_user
                ))
        else:
            log.warning(
                "{}_{} is already uploaded to repo '{}',"
                " environment '{}'".format(
                    changes['source'],
                    changes['version'],
                    repo,
                    p.env
                )
            )
            return None, None
    else:
        notification = notifications.new_changes(
            repo, env, changes['source'], time.time(), files, changes['changed-by']
        )
    pkg = p or package.Package.empty(repo)
    pkg.env = env
    pkg.source = changes['source']
    pkg.version = changes['version']
    affected_arch = set()
    for file in files:
        with open(file) as f:
            hashes = common.get_hashes(f)
        if checksums is not None:
            raw_storage_key, err = upload_deb(file, checksums[file][1], checksums[file][0])
        else:
            raw_storage_key, err = upload_deb(file)
        if err:
            return None, err
        storage_key = "/storage/{}".format(raw_storage_key)
        filename = os.path.basename(file)
        rbtorrent_id = None
        if repo_conf.generate_torrent and file.endswith(constants.deb_extensions):
            rbtorrent_id, err = do_generate_torrent(raw_storage_key, filename)
            if err:
                return None, err

        if file.endswith(constants.deb_extensions):
            doc = {
                'size': os.stat(file)[stat.ST_SIZE],
                'sha512': binary.Binary(hashes['sha512']),
                'sha256': binary.Binary(hashes['sha256']),
                'sha1': binary.Binary(hashes['sha1']),
                'md5': binary.Binary(hashes['md5']),
                'storage_key': storage_key
            }
            if repo_conf.generate_torrent and rbtorrent_id:
                doc['rbtorrent_id'] = rbtorrent_id
            try:
                deb = debfile.DebFile(file)
            except debfile.DebError as e:
                log.critical("Cannot load debfile %s: %s", file, e)
                return None, UploadPackageError(
                    "Cannot load debfile {0}: {1}".format(file, e))

            affected_arch.add(deb.debcontrol()['Architecture'])
            for k, v in deb.debcontrol().iteritems():
                doc[k] = v
            pkg.append_debs(doc)

        else:
            pkg.append_sources({
                'name': filename,
                'size': os.stat(file)[stat.ST_SIZE],
                'sha512': binary.Binary(hashes['sha512']),
                'sha256': binary.Binary(hashes['sha256']),
                'sha1': binary.Binary(hashes['sha1']),
                'md5': binary.Binary(hashes['md5']),
                'storage_key': storage_key
            })

            if file.endswith('.dsc'):
                with open(file) as f:
                    dsc = deb822.Dsc(f)
                    for k, v in dsc.iteritems():
                        if not k.startswith('Checksums-') and k != 'Files':
                            pkg.dsc[k] = v
    pkg.extend_audit(audit_meta)
    if affected_arch:
        # critical section. updating meta DB
        pkg.save()
        try:
            with common.RepoLock(repo, env):
                if skipUpdateMeta:
                    for arch in affected_arch:
                        index_store.set_dirty(repo, env, arch)
                        log.debug('Dirty bit for {}/{}/{} is set'.format(repo, env, arch))
                else:
                    for arch in affected_arch:
                        log.info(
                            "Updating '%s/%s/%s' repo metadata",
                            repo,
                            env,
                            arch)
                        update_repo_metadata(repo=repo, env=env, arch=arch)
        except UpdateRepoMetadataError as e:
            log.error("Error updating repo: %s", e)
            return None, UploadPackageError(
                "Failed to refresh some of Packages bunch file: {0}".format(e),
                retryable=True
            )
        except common.RepoLockTimeout as e:
            log.exception("Error updating repo {}, env {}".format(repo, env))
            return None, UploadPackageError(
                "Cannot lock repo {}, env {}: {}".format(repo, env, e),
                retryable=True
            )
        return notification, None
    else:
        log.info(
            "No changes made on repo %s/%s, skipping metadata update",
            repo,
            env)
    return None, None


class HashGetter(object):
    def __init__(self):
        self._md5 = self._md5_hex = self._sha1 = self._sha1_hex = None
        self._sha256 = self._sha256_hex = None

    def check_if_initialized(self):
        if not self._content:
            raise ValueError("Cannot get hash before content has been set!")

    def get_md5(self, hex=True):
        self.check_if_initialized()
        if (not self._md5) or (not self._md5_hex):
            md5 = hashlib.md5(self._content.getvalue())
            self._md5 = md5.digest()
            self._md5_hex = md5.hexdigest()
        if hex:
            return self._md5_hex
        return self._md5

    def get_sha1(self, hex=True):
        self.check_if_initialized()
        if (not self._sha1) or (not self._sha1_hex):
            sha1 = hashlib.sha1(self._content.getvalue())
            self._sha1 = sha1.digest()
            self._sha1_hex = sha1.hexdigest()
        if hex:
            return self._sha1_hex
        return self._sha1

    def get_sha256(self, hex=True):
        self.check_if_initialized()
        if (not self._sha256) or (not self._sha256_hex):
            sha256 = hashlib.sha256(self._content.getvalue())
            self._sha256 = sha256.digest()
            self._sha256_hex = sha256.hexdigest()
        if hex:
            return self._sha256_hex
        return self._sha256


class GenericIndexFile(object):
    def getDescriptor(self):
        return self._content

    def getSize(self):
        return self._len


class PlainIndexFile(GenericIndexFile, HashGetter):
    def __init__(self, index_file_producer, index_file_name):
        # We are getting values from DB in __init__, because parent get_md5,
        # get_sha1, get_sha256 methods may be called before getDescriptor
        super(self.__class__, self).__init__()

        self.filename = index_file_name
        self.storage_key = None

        data = common.myStringIO()
        index_file_producer(data)
        encoded = data.getvalue().encode('utf-8')
        data = common.myStringIO()
        data.write(encoded)

        self._len = data.tell()
        self._content = data


class GzipedIndexFile(GenericIndexFile, HashGetter):
    def __init__(self, plain_file, index_file_name):
        super(self.__class__, self).__init__()

        self.filename = "{}.gz".format(index_file_name)
        self.storage_key = None

        data = common.myStringIO()
        plain_file.seek(0)
        with gzip.GzipFile(fileobj=data, mode='w') as f:
            for chunk in plain_file:
                f.write(chunk)
        self._len = data.tell()
        self._content = data


class BzipedIndexFile(GenericIndexFile, HashGetter):
    def __init__(self, plain_file, index_file_name):
        super(self.__class__, self).__init__()

        self.filename = "{}.bz2".format(index_file_name)
        self.storage_key = None

        data = common.myStringIO()
        bz2Compressor = bz2.BZ2Compressor(1)
        plain_file.seek(0)
        for chunk in plain_file:
            compressed_chunk = bz2Compressor.compress(chunk)
            data.write(compressed_chunk)
        last_chunk = bz2Compressor.flush()
        data.write(last_chunk)
        self._len = data.tell()
        self._content = data


class IndexFileBunch(object):
    def __init__(self, index_file_producer, index_file_name):
        self.plain = PlainIndexFile(index_file_producer, index_file_name)
        self.gziped = GzipedIndexFile(self.plain._content, index_file_name)
        self.bziped = BzipedIndexFile(self.plain._content, index_file_name)


def generate_release_file(repo, env, arch, date, index_bunch, additional_fields=None):
    idx_nm = 'Sources' if arch == 'source' else 'Packages'
    release = u'Origin: {}\n'.format(repo)
    release += u'Suite: {}\n'.format(env)
    release += u'Codename: {}/{}\n'.format(env, arch)
    release += u'Date: {}\n'.format(date)
    release += u'Architectures: {}\n'.format(arch)
    if additional_fields:
        release += ''.join('{}: {}\n'.format(k, v) for k, v in additional_fields.items())
    release += u'Description: {}\n'.format(package_repository.Config.find(repo).description)

    release += u"MD5Sum:\n"
    release += " {}\t{} {}\n".format(
        index_bunch.plain.get_md5(), index_bunch.plain.getSize(), idx_nm)
    release += " {}\t{} {}.gz\n".format(
        index_bunch.gziped.get_md5(), index_bunch.gziped.getSize(), idx_nm)
    release += " {}\t{} {}.bz2\n".format(
        index_bunch.bziped.get_md5(), index_bunch.bziped.getSize(), idx_nm)

    release += u"SHA1:\n"
    release += " {}\t{} {}\n".format(
        index_bunch.plain.get_sha1(), index_bunch.plain.getSize(), idx_nm)
    release += " {}\t{} {}.gz\n".format(
        index_bunch.gziped.get_sha1(), index_bunch.gziped.getSize(), idx_nm)
    release += " {}\t{} {}.bz2\n".format(
        index_bunch.bziped.get_sha1(), index_bunch.bziped.getSize(), idx_nm)

    release += u"SHA256:\n"
    release += " {}\t{} {}\n".format(
        index_bunch.plain.get_sha256(), index_bunch.plain.getSize(), idx_nm)
    release += " {}\t{} {}.gz\n".format(
        index_bunch.gziped.get_sha256(), index_bunch.gziped.getSize(), idx_nm)
    release += " {}\t{} {}.bz2\n".format(
        index_bunch.bziped.get_sha256(), index_bunch.bziped.getSize(), idx_nm)
    return release


def gen_gpg_sigs(release):
    signer = common.config['gpg']['signer']
    release_data = release.encode('utf-8')
    release_gpg = common.gpg_sign(release_data, signer)
    in_release = common.gpg_sign_in_place(release_data, signer)
    return release_gpg, in_release


def generate_packages_file(repo, env, arch, data):
    pkgs = package.Package.find(repo, env=env, arch=arch)
    for pkg in pkgs:
        arch_debs = filter(
            lambda deb: deb['Architecture'] == arch,
            pkg.debs
        )
        for deb in arch_debs:
            for k, v in deb.iteritems():
                if k == 'md5':
                    string = "MD5sum: {0}\n".format(hexlify(v))
                elif k == 'sha1':
                    string = "SHA1: {0}\n".format(hexlify(v))
                elif k == 'sha256':
                    string = "SHA256: {0}\n".format(hexlify(v))
                elif k == 'sha512':
                    string = "SHA512: {0}\n".format(hexlify(v))
                elif k == 'storage_key':
                    string = "Filename: {0}\n".format(v)
                else:
                    string = u"{0}: {1}\n".format(
                        unicode(k.capitalize()),
                        unicode(v))
                data.write(string)
            data.write("\n")


def update_repo_metadata(repo=None, env=None, arch=None, force=False, index_store=package_index.default_store,
                         index_producer=generate_packages_file, index_type='Packages'):
    log.info("updating repo metadata: repo:{} env:{} arch:{} force:{}".format(
        repo, env, arch, force
    ))
    producer = partial(index_producer, repo, env, arch)
    old_repo = index_store.find_one(repo, env, arch)
    if old_repo and old_repo.force_index:
        force = True
    # commented out to avoid breaking existing code
    # if not force and old_repo and old_repo.dirty_bit is False:
    #     return
    index_bunch = IndexFileBunch(producer, index_type)

    plain_md5 = index_bunch.plain.get_md5()
    if old_repo and old_repo.plain:
        md5_matches = bool(old_repo.plain.find(plain_md5) >= 0)
    else:
        md5_matches = False
    if old_repo and old_repo.plain and md5_matches:
        log.warn(
            "%s file for %s/%s/%s not changed, skipping update",
            index_type, repo, env, arch)
        if not force:
            log.info('resetting dirty bit for repository: {}/{}/{}'.format(repo, env, arch))
            modified = index_store.set_clean(repo, env, arch).modified_count > 0
            if modified:
                log.debug("Dirty bit for {}/{} set at {} turning to false".format(
                    env, arch, datetime.now()))
            return
    plain = index_bunch.plain
    gzipped = index_bunch.gziped
    bzipped = index_bunch.bziped
    for index_file in (plain, gzipped, bzipped):
        # All hash functions requires calls to getDescriptor(), but we cannot
        # execute them on closed file-like myStringIO interface, so we will
        # call them here
        index_file.get_md5()
        index_file.get_sha1()
        index_file.get_sha256()

        base_key = "{}/{}/{}/{}_{}".format(
            repo, env, arch, index_file.filename, index_file.get_md5())

        if index_file.getSize() == 0:
            index_file.storage_key = constants.MDS_EMPTY_KEY
        else:
            index_file.storage_key = loader.get_plugin('storage').put(
                base_key, file=index_file.getDescriptor())

        if not index_file.storage_key:
            log.critical("Error uploading new index file", index_file.filename)
            raise UpdateRepoMetadataError(
                "Cannot upload {} file to storage".format(index_file.filename))

    # We don't need to generate Release file on-the-fly:
    # it's small enough to put it directly to metabase
    now = datetime.utcnow()
    release = generate_release_file(
        repo,
        env,
        arch,
        now.strftime("%a, %d %b %Y %H:%M:%S +0000"),
        index_bunch)
    release_gpg, in_release = gen_gpg_sigs(release)
    release_by_hash = generate_release_file(
        repo,
        env,
        arch,
        now.strftime("%a, %d %b %Y %H:%M:%S +0000"),
        index_bunch,
        {'Acquire-By-Hash': 'yes'}
    )
    release_gpg_by_hash, in_release_by_hash = gen_gpg_sigs(release_by_hash)
    if old_repo:
        idx = package_index.PackageIndex.from_index(old_repo)
    else:
        idx = package_index.PackageIndex(repo, arch=arch, env=env)
    idx.lastupdated = now
    idx.md5 = binary.Binary(index_bunch.plain.get_md5(hex=False))
    idx.sha1 = binary.Binary(index_bunch.plain.get_sha1(hex=False))
    idx.sha256 = binary.Binary(index_bunch.plain.get_sha256(hex=False))
    idx.size = index_bunch.plain.getSize()
    idx.release_file = release
    idx.in_release_file = in_release
    idx.release_gpg = release_gpg
    idx.plain = index_bunch.plain.storage_key
    idx.gzipped = index_bunch.gziped.storage_key
    idx.bzipped = index_bunch.bziped.storage_key
    idx.release_by_hash = release_by_hash
    idx.release_gpg_by_hash = release_gpg_by_hash
    idx.in_release_gpg_by_hash = in_release_by_hash
    idx.plain_sha256 = plain.get_sha256(hex=True).lower()
    idx.gzipped_sha256 = gzipped.get_sha256(hex=True).lower()
    idx.bzipped_sha256 = bzipped.get_sha256(hex=True).lower()
    idx.dirty_bit = False
    idx.dirty_bit_set_at = now
    index_store.save(idx)
    # schedule Sources update
    if arch != 'source':
        index_store.set_dirty(repo, env, 'source')


@callable_timing_hram(sources_daemon_indexing_time)
def update_sources_file(repo, env, force=False, index_store=package_index.default_store):
    def generate_sources_file(repo, env, _, src_file):
        def gen_meta(algo, files):
            for f in files:
                data = u" {} {} {}\n".format(
                    hexlify(f[algo]), f['size'], f['name']
                )
                src_file.write(data)

        def gen_src_record(pkg):
            for k, v in pkg.dsc.items():
                if k == 'Source':
                    src_file.write(u"Package: {0}\n".format(v))
                else:
                    src_file.write(u"{}: {}\n".format(k.capitalize(), v))
            src_file.write(u"Directory: {}/source\n".format(env))
            files = filter(
                lambda x: x['name'].endswith(constants.src_extensions),
                pkg.sources
            )

            src_file.write(u"Files: \n")
            gen_meta('md5', files)
            src_file.write(u"Checksums-Sha1: \n")
            gen_meta('sha1', files)
            src_file.write(u"Checksums-Sha256: \n")
            gen_meta('sha256', files)

            src_file.write(u"\n")

        pkgs = package.Package.find(repo, env=env, dsc={'$exists': True})
        map(gen_src_record, pkgs)
    return update_repo_metadata(repo, env, 'source', force=force, index_store=index_store,
                                index_producer=generate_sources_file, index_type='Sources')


def update_sources_files(interactive=False, index_store=package_index.default_store):
    global reg
    reg = common.RepoEnvsGetter()
    while True:
        sources_daemon_indexing_time.reset_measure()
        sources_daemon_indexing_time.start_measure()
        for repo in package_repository.list_all():
            for env in reg.get_envs(repo):
                dirty = index_store.find_one(repo, env, 'source', True)
                if dirty:
                    try:
                        with common.SrcLock(repo, env):
                            update_sources_file(repo, env)
                            msg = 'Sources for {}/{} has been sucsessfully' \
                                  ' updated'.format(repo, env)
                            log.info(msg)
                    except common.MetadataIsBusy:
                        msg = 'Sources file for {}/{} seems to be already ' \
                              'updating just right now.' \
                              ' Skipping'.format(repo, env)
                        log.debug(msg)
            sources_daemon_indexed_repo_count.put_value(1)
        sources_daemon_indexing_time.end_measure()
        dump_sources_daemon_stats(common.config['sources_daemon']['stats_file_path'])
        time.sleep(10)


class AllPackagesMetadataUpdater(object):
    def __init__(self):
        self._repos_queue = Queue()
        self._pipe_lock = Lock()
        self._not_indexing_repos_cache_last_update = None
        self.reag = common.RepoEnvArchesGetter()
        self.reg = common.RepoEnvsGetter()
        self.locks = {}
        index_cfg = common.config['indexer_daemon']
        self.processes_per_host = index_cfg['processes_per_host']
        self.repos_per_process = int(index_cfg['repos_per_host'] / self.processes_per_host)
        self.index_freq = index_cfg['index_freq']
        self.indexer_lock_timeout = index_cfg['indexer_lock_timeout']
        self.thread_start_freq = index_cfg['thread_start_freq']
        self.title_lock = Lock()
        self.thread_prefix = '__indexing_attempt_'

    def _extract_repo_names_from_locks(self, locks):
        repo_names = set()
        for lock in locks:
            small_lock_id = common.ZKLockFactory.extract_small_lock_id(lock)
            lock_id = common.DeferedIndexerLock.extract_clean_lock_id(
                small_lock_id)
            repo_names.add(lock_id)
        return repo_names

    def _get_contenders(self, repo):
        if repo not in self.locks:
            self.locks[repo] = common.DeferedIndexerLock(repo)
        _, oldest_node = AllPackagesMetadataUpdater._get_oldest_defered_lock(self.locks[repo])
        if oldest_node and time.time() - oldest_node[1].ctime / 1000.0 > self.indexer_lock_timeout:
            # return empty list if repo lock is too old
            return []
        try:
            contenders = self.locks[repo].contenders()
            return contenders
        except kazoo.exceptions.NoNodeError:
            return []

    def _get_indexing_repos(self, all_repos):
        # use only 8 threads for resolving indexing repos
        active_locks = []
        with ThreadPoolExecutor(8) as executor:
            for l in executor.map(self._get_contenders, all_repos):
                if not l:
                    continue
                active_locks.append(l[0])

        indexing_repos = self._extract_repo_names_from_locks(active_locks)
        return indexing_repos

    def _get_not_indexing_repos_bypassing_cache(self):
        all_repos = package_repository.list_all()
        indexing_repos = self._get_indexing_repos(all_repos)
        not_indexing_repos = set(all_repos) - indexing_repos
        self._not_indexing_repos_cache_last_update = datetime.now()
        return list(not_indexing_repos)

    def _get_not_indexing_repos(self):
        repos = self._get_not_indexing_repos_bypassing_cache()
        random.shuffle(repos)
        return repos

    def _update_metadata(self, repo, env, arch):
        try:
            with common.RepoLock(repo, env, timeout=600, nonblocking=False):
                update_repo_metadata(
                    repo=repo, env=env, arch=arch)
                msg = 'Packages files for {}/{}/{} has ' \
                      'been sucsessfully updated'.format(
                    repo, env, arch)
                log.info(msg)
        except common.MetadataIsBusy:
            msg = 'Pakages files for {}/{}/{} seems to ' \
                  'be already updating just right now.' \
                  ' Skipping'.format(repo, env, arch)
            log.debug(msg)

    def _get_dirty_metadatas(self, repo, index_store=package_index.default_store):
        for env in self.reg.get_envs(repo):
            arches = self.reag.get_arches(repo, env, skip_source=True)
            for arch in arches:
                dirty = index_store.find_one(repo, env, arch, True)
                if dirty:
                    msg = 'Metadata for {}/{}/{} is outdated, ' \
                          'will update it'.format(repo, env, arch)
                    log.debug(msg)
                    yield env, arch

    def _update_all_repo_metadata(self, repo):
        for env, arch in self._get_dirty_metadatas(repo):
            self._update_metadata(repo, env, arch)

    def _start_indexer_thread(self, repo):
        start_time = time.time()
        lock = common.DeferedIndexerLock(repo)
        try:
            with lock:
                self._update_all_repo_metadata(repo)
        except common.DeferedIndexerLockTimeout:
            log.warning('timed out trying to acquire lock on repo: {}'.format(repo))
            oldest_path, oldest_node = AllPackagesMetadataUpdater._get_oldest_defered_lock(lock)
            if oldest_node and time.time() - oldest_node[1].ctime / 1000.0 > self.indexer_lock_timeout:
                try:
                    lock.lock.client.delete(oldest_path, recursive=True)
                except Exception:
                    pass
            pass
        except (kazoo.handlers.threading.KazooTimeoutError, kazoo.exceptions.ConnectionLoss) as error:
            log.exception('zookeeper connection problem: {}'.format(error))
            pass
        except Exception as error:
            log.exception('Cannot index repo {}: {}'.format(repo, error))
            return None
        end_time = time.time()
        return end_time - start_time

    @staticmethod
    def _get_oldest_defered_lock(lock):
        zk = lock.lock.client
        try:
            lock_contender_nodes = zk.get_children(lock.lock.path)
        except kazoo.exceptions.NoNodeError:
            return None, None
        lock_contender_paths = map(
            lambda x, y: '{}/{}'.format(x, y),
            (lock.lock.path,) * len(lock_contender_nodes),
            lock_contender_nodes
        )
        oldest_node = None
        oldest_path = None
        for path in lock_contender_paths:
            try:
                node = zk.get(path)
            except kazoo.exceptions.NoNodeError:
                continue
            if oldest_node:
                if node[1].ctime < oldest_node[1].ctime:
                    oldest_node = node
                    oldest_path = path
            else:
                oldest_node = node
                oldest_path = path
        return oldest_path, oldest_node

    def _get_current_indexing_repos(self):
        threads_names = map(lambda x: x.name, threading.enumerate())
        threads_names = filter(
            lambda x: self.thread_prefix in x, threads_names)
        threads_names = map(
            lambda x: x.replace(self.thread_prefix, ''), threads_names)
        return threads_names

    def start_indexer_process(self, heartbeat_value):
        thread_pool = ThreadPoolExecutor(max_workers=self.repos_per_process)
        log.info('process {} spawned thread pool: {} threads'.format(multiprocessing.current_process().pid,
                                                                     self.repos_per_process))
        jobs = []
        jobs_scheduled = Accumulator('indexer_jobs_fetched_summ')
        jobs_done = Accumulator('indexer_jobs_done_summ')
        jobs_failed = Accumulator('indexer_jobs_failed_summ')
        job_timing_hgram = Histogram('indexer_job_time_ahhh')
        metrics_url = common.config['indexer_daemon']['stats_push_url']
        while True:
            job_done = False
            try:
                job_timing_hgram.start_measure()
                has_job = True
                try:
                    repo = self._repos_queue.get_nowait()
                except Exception:
                    log.debug('no repos available, sleeping 100ms')
                    time.sleep(0.1)
                    has_job = False
                if has_job:
                    log.debug(
                        'Starting indexing attempt thread for repo {}'.format(repo)
                    )
                    jobs.append(thread_pool.submit(self._start_indexer_thread, repo))
                    jobs_scheduled.put_value(1)
                for job in jobs:
                    if job.done():
                        job_done = True
                        jobs_done.put_value(1)
                        result = job.result()
                        if result is None:
                            jobs_failed.put_value(1)
                        else:
                            job_timing_hgram.put_value(result)
                        jobs.remove(job)
                log.debug('Current indexing {}/{} repos'.format(len(jobs), self._repos_queue.qsize()))
                setproctitle('packages-daemon-indexer: {}/{}'.format(len(jobs), self._repos_queue.qsize()))
                heartbeat_value.value = time.time()
                if job_done:
                    job_timing_hgram.end_measure()
                    metrics_pushed = True
                    metrics_pushed = metrics_pushed and push_metric(metrics_url, jobs_scheduled.to_json())
                    metrics_pushed = metrics_pushed and push_metric(metrics_url, jobs_done.to_json())
                    metrics_pushed = metrics_pushed and push_metric(metrics_url, jobs_failed.to_json())
                    job_timing_hgram_pushed = push_metric(metrics_url, job_timing_hgram.to_json())
                    metrics_pushed = metrics_pushed and job_timing_hgram_pushed
                    if not job_timing_hgram_pushed:
                        log.debug('job timing metric failed to push. will push next iteration')
                    if job_timing_hgram_pushed:
                        job_timing_hgram.reset_measure(force=True)
                    if metrics_pushed:
                        jobs_scheduled.reset_measure()
                        jobs_done.reset_measure()
                        log.debug('successfully pushed all metrics')
            except Exception as error:
                log.exception('error in worker process: {}'.format(error))
        exit(0)

    def start_indexer_manager(self):
        dead_time = common.config['indexer_daemon']['heartbeat_dead_time']
        fetcher_heartbeat = multiprocessing.Value('d', time.time())
        fetcher_process = Process(target=self.start_fetcher_process, args=(fetcher_heartbeat,))
        fetcher_process.daemon = True
        fetcher_process.start()
        indexer_processes = {}
        for i in range(0, self.processes_per_host):
            indexer_heartbeat = multiprocessing.Value('d', time.time())
            indexer = Process(target=self.start_indexer_process, args=(indexer_heartbeat,))
            indexer.daemon = True
            indexer.start()
            indexer_processes[i] = {
                'heartbeat': indexer_heartbeat,
                'process': indexer
            }
        while True:
            # Process is first checked (called waitpid), then checked for heartbeat time.
            # If process is dead or has been killed - we assign None and restart.
            try:
                if not fetcher_process.is_alive():
                    log.error('fetcher process is dead')
                    fetcher_process = None
                elif time.time() - fetcher_heartbeat.value > dead_time:
                    log.error('fetcher process {} last heartbeat {}, killing child'.format(
                        fetcher_process,
                        time.time() - fetcher_heartbeat.value
                    )
                    )
                    process.terminate_child(fetcher_process, 'fetcher')
                    fetcher_process = None
                if fetcher_process is None:
                    fetcher_process = Process(target=self.start_fetcher_process, args=(fetcher_heartbeat,))
                    fetcher_process.daemon = True
                    fetcher_process.start()
                    log.error('started new fetcher process with pid: {}'.format(fetcher_process.pid))
                else:
                    log.debug('fetcher process is alive')

                for pair in indexer_processes.values():
                    if not pair['process'].is_alive():
                        log.error('indexer process {} is dead'.format(pair['process'].pid))
                        pair['process'] = None
                    elif time.time() - pair['heartbeat'].value > dead_time:
                        log.error('indexer process {} last heartbeat {}, killing child'.format(
                            pair['process'].pid,
                            time.time() - pair['heartbeat'].value
                        )
                        )
                        process.terminate_child(pair['process'], 'indexer')
                        pair['process'] = None
                    if pair['process'] is None:
                        pair['process'] = Process(target=self.start_indexer_process, args=(pair['heartbeat'],))
                        pair['process'].daemon = True
                        pair['heartbeat'].value = time.time()
                        pair['process'].start()
                        log.error('started new indexer process with pid: {}'.format(pair['process'].pid))
                time.sleep(1)
            except:
                log.exception('error in manager process')

    def start_fetcher_process(self, heartbeat_value):
        log.info("initialized mongo connections(fetcher)")
        while True:
            try:
                log.debug('fetcher: fetching repo list')
                repos = self._get_not_indexing_repos()
                log.debug('fetcher: fetched repos: {}'.format(repos))
                for repo in repos:
                    log.debug('fetcher: queuing repo: {}'.format(repo))
                    self._repos_queue.put(repo)
                setproctitle('packages-daemon-fetcher: {}/{}'.format(len(repos), self._repos_queue.qsize()))
                heartbeat_value.value = time.time()
                time.sleep(self.index_freq)
            except (kazoo.handlers.threading.KazooTimeoutError, kazoo.exceptions.ConnectionLoss) as error:
                log.error('zookeeper connection problem')
                log.error(error)
            except Exception as error:
                log.error('got unexpected error scheduling repo queue')
                log.error(error)
        exit(0)


def update_all_packages_metadata():
    log.info('Starting packages daemon')
    allPackagesMetadataUpdater = AllPackagesMetadataUpdater()
    allPackagesMetadataUpdater.start_indexer_manager()


def clean_branch(repo=None, branch=None):
    if not branch.endswith("2remove"):
        return {'result': constants.Status.ERROR, 'msg': "Branch name have to ends with '2remove'"}
    try:
        with common.RepoLock(repo, branch):
            result = package.Package.find(repo, env=branch)
            if not result:
                msg = "Cannot find branch {} in repo '{}'".format(branch, repo)
                log.error(msg)
                return {'result': constants.Status.NOT_FOUND, 'msg': msg}

            affected_arch = set()
            log.info("preparing to delete {} packages from MDS".format(len(result)))
            for pkg in result:
                if pkg.sources:
                    results = pkg.debs + pkg.sources
                else:
                    results = pkg.debs

                for storage_key in results:
                    if 'Architecture' in storage_key:
                        affected_arch.add(storage_key['Architecture'])
                        log.info("Removing unnecesary deb file {}".format(storage_key['storage_key']))
                    else:
                        log.info("Removing unnecesary source file {}".format(storage_key['storage_key']))
                    dirty_key = storage_key['storage_key']
                    clean_key = loader.get_plugin('storage').cleanify_url(dirty_key)
                    loader.get_plugin('storage').delete(clean_key)

            delete_result = package.Package.delete_many(repo, env=branch)
            log.info(
                "deleted {} packages from DB ack {}".format(delete_result.deleted_count, delete_result.acknowledged)
            )

            for arch in affected_arch:
                log.info("Updating '{}/{}/{}' repo metadata".format(repo, branch, arch))
                update_repo_metadata(repo=repo, env=branch, arch=arch)

            msg = "Packages were removed from repo '{}' branch {}".format(repo, branch)
            log.info(msg)
            return {'result': constants.Status.OK, 'msg': msg}

    except (common.RepoLockTimeout, UpdateRepoMetadataError) as e:
        msg = "Rmove failed: {}".format(e)
        return {'result': constants.Status.TIMEOUT, 'msg': msg}


def recycle_branch(min_recycle_period, repo, branch, recycle_after):
    if not branch.endswith("2remove"):
        return {'result': constants.Status.ERROR, 'msg': "Branch name have to ends with '2remove'"}
    try:
        with common.RepoLock(repo, branch):
            result = package.Package.find(repo, env=branch, recycle_after={'$lt': recycle_after})
            if not result:
                msg = "Cannot find branch {} in repo '{}'".format(branch, repo)
                log.error(msg)
                return {'result': constants.Status.NOT_FOUND, 'msg': msg}

            affected_arch = set()
            log.info("preparing to delete {} packages from MDS".format(len(result)))
            for pkg in result:
                e = audit.get_last_event(pkg, 'dmove')
                if e:
                    recycle_period = (pkg.recycle_after - e['timestamp']).total_seconds()
                    if recycle_period < min_recycle_period:
                        log.error("Cannot recycle package {}={} in {}/{}: recycle period too short".format(
                            pkg.source, pkg.version, repo, branch
                        ))
                        continue
                else:
                    log.error(
                        "Package {}={} in {}/{} has recycle_after but was not dmoved to recycle bin properly".format(
                            pkg.source, pkg.version, repo, branch
                        ))
                    continue
                results = pkg.debs
                if pkg.sources:
                    results.extend(pkg.sources)
                for storage_key in results:
                    if 'Architecture' in storage_key:
                        affected_arch.add(storage_key['Architecture'])
                        log.info("Removing unnecessary deb file {}".format(storage_key['storage_key']))
                    else:
                        log.info("Removing unnecessary source file {}".format(storage_key['storage_key']))
                    dirty_key = storage_key['storage_key']
                    clean_key = loader.get_plugin('storage').cleanify_url(dirty_key)
                    loader.get_plugin('storage').delete(clean_key)

                delete_result = pkg.delete()
                log.info(
                    "Package {}={} in {}/{} deleted {} packages from DB ack {}".format(
                        pkg.source, pkg.version, repo, branch,
                        delete_result.deleted_count, delete_result.acknowledged)
                )
            for arch in affected_arch:
                log.info("Updating '{}/{}/{}' repo metadata".format(repo, branch, arch))
                update_repo_metadata(repo=repo, env=branch, arch=arch)

            msg = "Packages were removed from repo '{}' branch {}".format(repo, branch)
            log.info(msg)
            return {'result': constants.Status.OK, 'msg': msg}

    except (common.RepoLockTimeout, UpdateRepoMetadataError) as e:
        msg = "Rmove failed: {}".format(e)
        return {'result': constants.Status.TIMEOUT, 'msg': msg}


def dmove_package(pkg=None, version=None, repo=None, source=None, destination=None, skipUpdateMeta=False,
                  use_zk_locks=True, recycle_after=None, index_store=package_index.default_store):
    # TODO: exit if some of necessary parameter are missing

    # TODO: Think about what we must to do if master flaps between
    # find_and_modify and update_repo_metadata or between two
    # update_repo_metadata calls

    # set up dummy locks for using in repo-cleaner.apply
    if not use_zk_locks:
        src_lock = threading.Lock()
        dst_lock = threading.Lock()
    else:
        src_lock = common.RepoLock(repo, source)
        dst_lock = common.RepoLock(repo, destination)
    try:
        with src_lock:
            with dst_lock:
                audit_record = {
                    'timestamp': datetime.utcnow(),
                    'event': 'dmove',
                    'repo': repo,
                    'src': source,
                    'dst': destination,
                    'user': os.getenv('SUDO_USER', 'root'),
                }
                p = package.Package.find_one(repo, env={'$in': [source, destination]}, source=pkg, package=pkg, version=version)
                if not p:
                    msg = "Cannot find package '{}_{}' in repo '{}'" \
                          " at env {}".format(pkg, version, repo, source)
                    log.error(msg)
                    return {'result': constants.Status.NOT_FOUND, 'msg': msg}
                elif p.env == destination:
                    msg = "Package '{}_{}' is already in repo '{}'" \
                          " at env {}".format(pkg, version, repo, source)
                    log.warning(msg)
                    return {'result': constants.Status.NO_CHANGES, 'msg': msg}

                p.append_audit(audit_record)
                p.env = destination
                if destination.endswith('2remove') and recycle_after:
                    p.recycle_after = recycle_after
                result = p.save()

                if result.modified_count == 1:
                    msg = "Package '{}_{}' was dmoved in repo '{}'" \
                          " from {} to {}".format(p.source,
                                                  version,
                                                  repo,
                                                  source,
                                                  destination)
                    log.info(msg)
                else:
                    msg = "Package '{}_{}' dmove failed in repo '{}'" \
                          " from {} to {}".format(p.source,
                                                  version,
                                                  repo,
                                                  source,
                                                  destination)
                    log.error(msg)
                affected_arch = set()
                for d in p.debs:
                    affected_arch.add(d['Architecture'])
                for arch in affected_arch:
                    if not skipUpdateMeta:
                        log.info(
                            "Updating '%s/%s/%s' repo metadata",
                            repo, source, arch)
                        update_repo_metadata(repo=repo, env=source, arch=arch)
                        log.info(
                            "Updating '%s/%s/%s' repo metadata",
                            repo, destination, arch)
                        update_repo_metadata(repo=repo,
                                             env=destination,
                                             arch=arch)
                    else:
                        index_store.set_dirty(repo, source, arch)
                        index_store.set_dirty(repo, destination, arch)
                return {'result': constants.Status.OK, 'msg': msg, 'affected_archs': affected_arch}
    except (common.RepoLockTimeout, UpdateRepoMetadataError) as e:
        msg = "Dmove failed: {}".format(e)
        return {'result': constants.Status.TIMEOUT, 'msg': msg}


def bmove_package(pkg=None, version=None, source=None, destination=None, env=None, skipUpdateMeta=False,
                  delete=True, index_store=package_index.default_store):
    # TODO: exit if some of necessary parameterer are missing

    # TODO: Think about what we must to do if master flaps between
    # find_and_modify and update_repo_metadata or between two
    # update_repo_metadata calls

    if delete:
        lock = contextlib.nested(
            common.RepoLock(source, env),
            common.RepoLock(destination, env)
        )
    else:
        lock = common.RepoLock(destination, env)

    try:
        with lock:
            p = package.Package.find_one(source, env=env, source=pkg, version=version, package=pkg)
            if not p:
                msg = "Cannot find package '{}_{}' in repo '{}'" \
                      " at env {}".format(pkg, version, source, env)
                log.error(msg)
                return {'result': constants.Status.NOT_FOUND, 'msg': msg}

            file_re = r"/storage/(?P<grp_id>\d+)/(?P<repo>[-.\w]+)/(?P<pkg>.*)"
            file_re = re.compile(file_re)
            affected_arch = set()

            # move debs
            for index, deb in enumerate(p.debs):
                affected_arch.add(deb['Architecture'])
                storage_match = file_re.match(deb["storage_key"])
                old_key = "{}/{}/{}".format(
                    storage_match.group("grp_id"),
                    storage_match.group("repo"),
                    storage_match.group("pkg")
                )
                new_key = "{}/{}".format(
                    destination,
                    storage_match.group("pkg")
                )

                if delete:
                    storage_key = loader.get_plugin('storage').rename(
                        old_key, new_key)
                else:
                    storage_key = loader.get_plugin('storage').copy(
                        old_key, new_key)
                p.debs[index]["storage_key"] = "/storage/{}".format(
                    storage_key)

            # move sources
            if p.sources:
                for index, src in enumerate(p.sources):
                    storage_match = file_re.match(src["storage_key"])
                    old_key = "{}/{}/{}".format(
                        storage_match.group("grp_id"),
                        storage_match.group("repo"),
                        storage_match.group("pkg")
                    )
                    new_key = "{}/{}".format(
                        destination,
                        storage_match.group("pkg")
                    )

                    if delete:
                        storage_key = loader.get_plugin('storage').rename(
                            old_key, new_key)
                    else:
                        storage_key = loader.get_plugin('storage').copy(
                            old_key, new_key)
                    full_storage_key = "/storage/{}".format(storage_key)
                    p.sources[index]["storage_key"] = full_storage_key
            np = package.Package.from_package(p)
            np.repo = destination
            np.save()
            if delete:
                p.delete()

            for arch in affected_arch:
                if not skipUpdateMeta:
                    if delete:
                        log.info(
                            "Updating '%s/%s/%s' repo metadata",
                            source, env, arch)
                        update_repo_metadata(repo=source, env=env, arch=arch)
                    log.info(
                        "Updating '%s/%s/%s' repo metadata",
                        destination, env, arch)
                    update_repo_metadata(repo=destination, env=env, arch=arch)
                else:
                    if delete:
                        index_store.set_dirty(source, env, arch)
                    index_store.set_dirty(destination, env, arch)
            if delete:
                operation = "bmove"
            else:
                operation = "bcopy"
            msg = "Package '{}_{}' was '{}'ed into repo '{}' env '{}'" \
                  " from {} env {}".format(np.source,
                                           version,
                                           operation,
                                           destination,
                                           env,
                                           source,
                                           env)
            log.info(msg)

            return {'result': constants.Status.OK, 'msg': msg}

    except (common.RepoLockTimeout, UpdateRepoMetadataError) as e:
        msg = "{} failed: {}".format(operation, e)
        return {'result': constants.Status.TIMEOUT, 'msg': msg}


def rmove_package(pkg=None, version=None, repo=None, env=None, skipUpdateMeta=False, index_store=package_index.default_store):
    try:
        with common.RepoLock(repo, env):
            p = package.Package.find_one(repo, env=env, source=pkg, version=version, package=pkg)
            if not p:
                msg = "Cannot find package '{}_{}' in repo '{}'" \
                      " at env {}".format(pkg, version, repo, env)
                log.error(msg)
                return {'result': constants.Status.NOT_FOUND, 'msg': msg}

            affected_arch = set()

            if p.sources:
                results = p.sources + p.debs
            else:
                results = p.debs

            for storage_key_container in results:
                if 'Architecture' in storage_key_container:
                    affected_arch.add(storage_key_container['Architecture'])
                    log.debug("Removing unnecesary deb file {}".format(
                        storage_key_container['storage_key']))
                else:
                    log.debug("Removing unnecesary source file {}".format(
                        storage_key_container['storage_key']))
                dirty_key = storage_key_container['storage_key']
                clean_key = loader.get_plugin('storage').cleanify_url(
                    dirty_key
                )
                loader.get_plugin('storage').delete(clean_key)
            p.delete()
            for arch in affected_arch:
                if not skipUpdateMeta:
                    log.info(
                        "Updating '%s/%s/%s' repo metadata", repo, env, arch)
                    update_repo_metadata(repo=repo, env=env, arch=arch)
                else:
                    index_store.set_dirty(repo, env, arch)
            msg = "Package '{}_{}' was removed from repo '{}'" \
                  " env {}".format(p.source, version, repo, env)
            log.info(msg)
            return {'result': constants.Status.OK, 'msg': msg}

    except (common.RepoLockTimeout, UpdateRepoMetadataError) as e:
        msg = "Rmove failed: {}".format(e)
        return {'result': constants.Status.TIMEOUT, 'msg': msg}


def exec_defered_rmove(deleted_key_store=deleted_key.default_store):
    for failed_key in deleted_key_store.all():
        log.info('Key {} was not properly deleted, '
                 'retrying to delete it.'.format(failed_key))
        err = loader.get_plugin('storage').delete(failed_key)
        if not err:
            deleted_key_store.pop(failed_key)
        else:
            log.error('Cannot delete MDS key: {}'.format(err))


def add_repository(repo=None, descr=None):
    if not repo:
        raise ValueError('Cannot create repo without name!')
    if repo in constants.reserved_collection_names:
        raise ValueError(
            'Names {} are reserved.'.format(constants.reserved_collection_names)
        )
    inc_template = common.config['duploader_daemon']['incoming_dir_template']
    incoming_dir = inc_template.format(repo)
    root_template = common.config['duploader_daemon']['repo_root_template']
    repo_root = root_template.format(repo)
    blank_descr_msg = 'Do not forget description'
    package_repository.add(repo, description=descr or blank_descr_msg, incoming_dir=incoming_dir, repo_root=repo_root,
                           update_sources=update_sources_file, update_packages=update_repo_metadata)


# TODO: remove unused feature
def mask_repository(repo=None):
    reg = common.RepoEnvsGetter()
    envs = reg.get_envs(repo)
    lock_list = []
    for env in envs:
        lock = common.RepoLock(repo, env)
        lock_list.append(lock)

    if not lock_list:
        msg = "Repository '{}' seems to be empty. " \
              "Nothong to remove.".format(repo)
        log.error(msg)
        return {'result': constants.Status.NOT_FOUND, 'msg': msg}

    combined_lock = contextlib.nested(*lock_list)
    try:
        with combined_lock:
            mongo_connection.repos()[repo].rename("__" + repo + "_masked__")
            mongo_connection.cacus()[repo].rename("__" + repo + "_masked__")
    except (common.RepoLockTimeout, UpdateRepoMetadataError) as e:
        msg = "Rmove failed: {}".format(e)
        return {'result': constants.Status.TIMEOUT, 'msg': msg}


def rmove_repository(repo=None, index_store=package_index.default_store):
    # get list of all packages and lists
    index_bunches = index_store.all(repo)
    all_packages = package.Package.find(repo)

    # drop all indexes
    def _delete_key(k):
        if k and k != constants.MDS_EMPTY_KEY:
            log.debug("Removing index file %s", k)
            loader.get_plugin('storage').delete(k)

    for idx in index_bunches:
        _delete_key(idx.plain)
        _delete_key(idx.gzipped)
        _delete_key(idx.bzipped)
        index_store.delete(idx)
    index_store.drop(repo)

    # exec rmove on each package
    for p in all_packages:
        rmove_package(
            pkg=p.source,
            version=p.version,
            repo=repo, env=p.env,
            skipUpdateMeta=True
        )

    log.info('Successfully removed repository "{}"'.format(repo))


def list_repos():
    for repo in package_repository.list_all():
        print(repo)
