from __future__ import absolute_import
from infra.dist.cacus.lib.stats.signal import Signal
import json


class Histogram(Signal):
    MAX_BUCKETS = 50

    def __init__(self, name):
        Signal.__init__(self, name)
        self.min = None
        self.max = None
        self.measuring = False
        self.values_read = False
        self.__values = []
        self._has_values = False

    def start_measure(self):
        if not self.measuring:
            self.measuring = True
            if self.values_read:
                self.__values = []
                self.values_read = False

    def end_measure(self):
        self.measuring = False

    def read_values(self):
        self.__values = sorted(self.__values)
        buckets = {}
        if self._has_values:
            bucket_size = float(self.max - self.min) / float(Histogram.MAX_BUCKETS)
            if bucket_size == 0:
                bucket_size = 1.0
            for v in self.__values:
                bucket = int((v - self.min) / bucket_size)
                buckets[bucket] = buckets.get(bucket, 0.0) + 1
        self.values_read = True
        measure_buckets = {}
        for i in buckets.keys():
            left_measure_border = self.min + i * bucket_size
            measure_buckets[left_measure_border] = buckets[i]
        return measure_buckets

    def reset_measure(self, force=False):
        if self.values_read or force:
            self.min = None
            self.max = None
            self.values_read = False
            self.__values = []
            self._has_values = False
        self.measuring = False

    def put_value(self, value):
        self.__values.append(value)
        if self.min is not None or self.max is not None:
            if value < self.min:
                self.min = value
            if value > self.max:
                self.max = value
        if self.max is None and self.min is None:
            self.max = value
            self.min = value
        self._has_values = True

    def has_values(self):
        return self._has_values

    def zero_value(self):
        return [0, 0]

    def get_value(self):
        return self.__values

    @classmethod
    def from_json(cls, json_string):
        json_repr = json.loads(json_string)
        obj = Histogram(json_repr['name'])
        obj.__values = json_repr['value']
        obj._has_values = True
        obj.min = min(json_repr['value'])
        obj.max = max(json_repr['value'])
