from __future__ import absolute_import

import collections
import datetime
import logging
import re
import time

import yaml

import infra.dist.cacus.lib.common
import infra.dist.cacus.lib.repo_manage
from infra.dist.cacus.lib import constants
from infra.dist.cacus.lib.dbal import package
from infra.dist.cacus.lib.dbal import package_repository
from infra.dist.cacus.lib.tools.repo_cleaner.package_filter_factory import PackageFilterFactory
from infra.dist.cacus.lib.utils.version import Version

DEFAULT_RECYCLE_PERIOD = 31 * 24 * 3600
DEFAULT_MIN_RECYCLE_PERIOD = 30 * 24 * 3600


def load_cleaner_config(log, path):
    try:
        with open(path, 'r') as f:
            config = yaml.load(f)
    except Exception as error:
        log.critical('error loading repo-cleaner config: %s', path)
        log.critical(error)
        exit(-1)
    log.debug('loaded config: %s', config)
    return config


def prepare_batch(args):
    log = logging.getLogger('infra.dist.cacus.lib.tools.repo-cleaner.prepare')
    config = load_cleaner_config(log, args.repos_config)
    filter_factory = PackageFilterFactory(config)
    if 'all' in args.repo:
        repos = package_repository.list_all()
    else:
        repos = args.repo
    log.info('collecting packages to clean in repos: %s', repos)
    for repo in repos:
        recycle_period = get_recycle_period_from_config(config, repo)
        packages_to_remove = collections.defaultdict(list)
        log.info('collecting packages in repo: %s', repo)
        packages = package.Package.find(repo, env={'$not': re.compile('^.*2remove$')})
        packages_with_version = map(
            lambda x: {
                'Version': Version.from_string(x.version),
                'Source': x.source,
                'environment': x.env,
                'audit_meta': x.audit_meta,
            },
            packages
        )
        del packages
        packages_with_version = sorted(
            packages_with_version,
            key=lambda k: (k['environment'], k['Source'], k['Version']),
            reverse=True
        )
        log.debug('packages in order:')
        for pkg in packages_with_version:
            log.debug('pkg: %s %s %s %s', repo, pkg['environment'], pkg['Source'], pkg['Version'])

        for pkg in packages_with_version:
            name = pkg['Source']
            version = pkg['Version']
            env = pkg['environment']
            btime = time.time()
            if 'audit_meta' in pkg and pkg['audit_meta'] is not None:
                for event in pkg['audit_meta']:
                    if event['event'] == 'upload':
                        ts = event['timestamp']
                        pkg_time = time.mktime(ts.timetuple())
                        if pkg_time < btime:
                            btime = pkg_time
            log.debug('got package: %s=%s(%s), btime: %s', name, version, env, btime)
            flt = filter_factory.get_filter(repo, env, name)
            if flt.filter(repo, env, name, version, btime):
                log.debug('package: %s=%s(%s), btime: %s scheduled to remove', name, version, env, btime)
                packages_to_remove[str(env)].append({
                    'package': str(name),
                    'version': str(version)
                })
        with open('{}/{}.yaml'.format(args.output_dir, repo), 'w') as f:
            d = {
                'recycle_period': recycle_period
            }
            if len(packages_to_remove) > 0:
                d['environments'] = dict(packages_to_remove)
            yaml.dump({str(repo): d}, f, default_flow_style=False)


def apply_batch(args):
    log = logging.getLogger('infra.dist.cacus.lib.tools.repo-cleaner.apply')
    for batch in args.batch:
        log.info('loading batch: %s', batch)
        with open(batch, 'r') as f:
            repos = yaml.load(f)
        for repo in repos.keys():
            affected_archs = set()
            recycle_period = datetime.timedelta(seconds=int(repos.get('recycle_period', DEFAULT_RECYCLE_PERIOD)))
            for src_env in repos[repo].get('environments', []):
                dst_env = '{}2remove'.format(src_env)
                with infra.dist.cacus.lib.common.RepoLock(repo, src_env):
                    with infra.dist.cacus.lib.common.RepoLock(repo, dst_env):
                        for p in repos[repo]['environments'][src_env]:
                            name = p['package']
                            version = p['version']
                            log.info('moving package %s=%s from %s to %s', name, version, src_env, dst_env)
                            if args.yes_i_really_want_lsr_ticket:
                                recycle_after = datetime.datetime.utcnow() + recycle_period
                                result = infra.dist.cacus.lib.repo_manage.dmove_package(name, version, repo, src_env,
                                                                                        dst_env, True, False,
                                                                                        recycle_after)
                                log.info(result['msg'])
                                if result['result'] == constants.Status.OK:
                                    log.info('moving package %s=%s from %s to %s: SUCCESS', name, version, src_env,
                                             dst_env)
                                    log.debug('affected_archs: %s', result['affected_archs'])
                                    affected_archs.update(result['affected_archs'])
                                else:
                                    if not args.keep_going:
                                        log.critical('dmove failed... exiting')
                                        exit(-1)
                                    else:
                                        log.warning('moving package %s=%s from %s to %s: FAILED', name, version,
                                                    src_env, dst_env)
                        for arch in affected_archs:
                            log.info('updating %s/%s/%s metadata', repo, src_env, arch)
                            infra.dist.cacus.lib.repo_manage.update_repo_metadata(repo, src_env, arch, True)
                            log.info('updating %s/%s/%s metadata', repo, dst_env, arch)
                            infra.dist.cacus.lib.repo_manage.update_repo_metadata(repo, dst_env, arch, True)


def get_min_recycle_period_from_config(config, repo):
    default = config.get('defaults', {}).get('min_recycle_period', DEFAULT_MIN_RECYCLE_PERIOD)
    return config.get('repos', {}).get(repo, {}).get('min_recycle_period', default)


def get_recycle_period_from_config(config, repo):
    default = config.get('defaults', {}).get('recycle_period', DEFAULT_RECYCLE_PERIOD)
    return config.get('repos', {}).get(repo, {}).get('recycle_period', default)


def recycle_packages(args):
    log = logging.getLogger('infra.dist.cacus.lib.tools.repo-cleaner.recycle')
    config = load_cleaner_config(log, args.repos_config)
    if 'all' in args.repo:
        repos = package_repository.list_all()
    else:
        repos = args.repo
    log.info('collecting packages to recycle in repos: %s', repos)
    for repo in repos:
        log.info('scanning repo %s', repo)
        min_recycle_period = get_min_recycle_period_from_config(config, repo)
        recycle_after = datetime.datetime.utcnow()
        for i in package_repository.find_recycle_branches(repo, recycle_after):
            r = infra.dist.cacus.lib.repo_manage.recycle_branch(min_recycle_period, repo, i, recycle_after)
            if r['result'] != constants.Status.OK:
                log.critical('cannot recycle repo: %s branch: %s', repo, i['_id'])
            log.info('Recycled packages: {}'.format(str(r)))


def main(args):
    log = logging.getLogger('infra.dist.cacus.lib.tools.repo-cleaner')
    log.info('starting repo-cleaner action: %s', args.sub_action)
    log.debug('args: %s', vars(args))
    if args.sub_action == 'prepare':
        prepare_batch(args)
    elif args.sub_action == 'apply':
        apply_batch(args)
    elif args.sub_action == 'recycle':
        recycle_packages(args)
