from tempfile import mkstemp
import subprocess
import os
import time
import logging

from infra.dist.cacus.lib.utils.staff import StaffFetcher, StaffError
from infra.dist.cacus.lib import common
from infra.dist.cacus.lib.daemon.gpg import stats as gpg_daemon
from infra.dist.cacus.lib.dearmor import dearmor_key

log = logging.getLogger(__name__)


MAIN_KEYRING = 'main_keyring'


def get_keys():
    headers = {
        "Accept": "application/json",
        "Authorization": common.config['gpg']['token'],
    }
    read_url = common.config['gpg']['read_url']
    params = common.config['gpg']['read_params']

    fetcher = StaffFetcher(read_url, params, headers)
    try:
        people = fetcher.get_all()
    except StaffError:
        log.exception("cannot fetch people from staff")
        return None

    keys = []
    for person in people:
        for key in person['gpg_keys']:
            keys.append((person['login'], key['key']))

    gpg_daemon.stats['gpg_daemon_keys_fetch_count_attt'] = len(keys)
    return keys


def get_groups():
    center_groups = common.config['gpg']['mapping'].keys()

    headers = {
        "Accept": "application/json",
        "Authorization": common.config['gpg']['token']
    }
    read_url = common.config['gpg']['groups']['read_url']
    params = common.config['gpg']['groups']['read_params']

    groups = {}

    for group in center_groups:
        params['group.url'] = group
        fetcher = StaffFetcher(read_url, params, headers)
        result = fetcher.get_all()
        groups[group] = {g['person']['login'] for g in result}

    return groups


def group_keys(keys, groups):
    keyrings = {}
    keyrings[MAIN_KEYRING] = []

    for login, key in keys:
        for group, logins in groups.items():
            if login in logins:
                if group not in keyrings:
                    keyrings[group] = []
                keyrings[group].append(key)

        keyrings[MAIN_KEYRING].append(key)

    return keyrings


def generate_raw_keyrings(key_groups):
    keyrings = {}

    for group, keys in key_groups.items():
        for key in keys:
            log.debug("dearmoring key {}".format(key))
            if group not in keyrings:
                keyrings[group] = bytearray()
            dearmored_key = dearmor_key(key)
            keyrings[group] += bytearray(dearmored_key)

    return keyrings


def get_gpg_keyrings():
    keyrings = {}
    fetch_start = time.time()

    keys = get_keys()
    groups = get_groups()
    key_groups = group_keys(keys, groups)
    raw_keyrings = generate_raw_keyrings(key_groups)

    fetch_end = time.time()
    gpg_daemon.stats['gpg_daemon_keys_fetch_time_axxx'] = fetch_end - fetch_start

    keyrings_size_sum = 0

    for keyring in raw_keyrings.values():
        keyrings_size_sum += len(keyring)

    gpg_daemon.stats['gpg_daemon_keys_fetch_size_attt'] = keyrings_size_sum

    log.info('loaded {} keyrings'.format(len(raw_keyrings)))

    for group, keyring in raw_keyrings.items():
        fd, tmppath = mkstemp()
        tmpfile = open(tmppath, 'wb')
        tmpfile.write(keyring)
        keyrings_size_sum += tmpfile.tell()
        tmpfile.close()
        os.close(fd)
        p = subprocess.Popen(
            [
                'gpg',
                '--no-default-keyring',
                '--keyring',
                tmppath,
                '--update-trustdb'],
            stdout=subprocess.PIPE
        )
        p.communicate()
        tmpfile = open(tmppath, 'rb')
        keyrings[group] = bytearray(tmpfile.read())
        tmpfile.close()
        os.unlink(tmppath)
        os.unlink(tmppath + "~")

    return keyrings
