from collections import namedtuple
import hashlib
import os


FilesEntry = namedtuple('FilesEntry', ['md5', 'size', 'section', 'priority', 'filename'])
SHA1Entry = namedtuple('SHA1Entry', ['sha1', 'size', 'filename'])
SHA256Entry = namedtuple('SHA256Entry', ['sha256', 'size', 'filename'])


def readln(f):
    line = f.readline()
    if line:
        return line.rstrip()
    else:
        return None


def get_sha256_and_size(path):
    h = hashlib.sha256()
    with open(path, 'rb') as f:
        d = f.read(4096)
        while d:
            h.update(d)
            d = f.read(4096)
        return h.hexdigest(), f.tell()


class ChangeFileException(Exception):
    pass


class ChangeFile(object):
    def __init__(self):
        self.filename = None
        self._data = None

    def load_from_file(self, path):
        """
        Loads changes from changes file at <path>.
        Look https://man7.org/linux/man-pages/man5/deb-changes.5.html for changes format spec.
        path: path to changes file
        """
        self.filename = path
        self._data = {}
        with open(self.filename, 'r') as f:
            k, v = None, None
            line = readln(f)
            while line is not None:
                if 'BEGIN PGP SIGNED MESSAGE' in line or line == '':
                    line = readln(f)
                    continue
                if 'BEGIN PGP SIGNATURE' in line:
                    break
                # field begin
                if line[0] != ' ':
                    if k:
                        self._data[k] = v
                    parts = line.split(":", 1)
                    if len(parts) != 2:
                        raise ChangeFileException('malformed line: "{}" in file: "{}"'.format(line, self.filename))
                    k, v = parts
                    k = k.lower()
                    # special typed fields
                    if k in ('files', 'checksums-sha1', 'checksums-sha256'):
                        v = []
                    else:
                        v = v.strip()
                else:
                    # '\n' replaced with ' .'
                    if line == ' .':
                        v += '\n'
                    # typed fields
                    elif k == 'files':
                        v.append(FilesEntry(*line.split()))
                    elif k == 'checksums-sha1':
                        v.append(SHA1Entry(*line.split()))
                    elif k == 'checksums-sha256':
                        v.append(SHA256Entry(*line.split()))
                    # regular multiline field
                    else:
                        v += line.strip()
                line = readln(f)
            self._data[k] = v

    def __getitem__(self, item):
        return self._data[item.lower()]

    def __contains__(self, item):
        return item.lower() in self._data

    def get_files(self):
        return [(x.md5, int(x.size), x.filename) for x in self._data['files']]

    def __str__(self):
        return str(self._data)

    def __repr__(self):
        return self._data.__repr__()

    def verify(self, path, hash_func=get_sha256_and_size):
        errors = []
        for file_info in self._data['checksums-sha256']:
            check_path = os.path.join(path, file_info.filename)
            try:
                sha256_sum, sz = hash_func(check_path)
                if sz != int(file_info.size):
                    errors.append('file {} size differs: changes size: {}, actual size: {}'.format(
                        file_info.filename, file_info.size, sz
                    ))
                if sha256_sum != file_info.sha256:
                    errors.append('file {} sha256 differs: changes sha256: {}, actual sha256: {}'.format(
                        file_info.filename, file_info.sha256, sha256_sum
                    ))
            except OSError as e:
                # report os errors for all files
                # helps to detect partial upload errors
                errors.append('error opening file {}: {}'.format(file_info.filename, e))
        if errors:
            raise ChangeFileException(', '.join(errors))
