import requests
import logging


log = logging.getLogger(__name__)


class StaffError(Exception):
    pass


class StaffFetcher(object):
    def __init__(self, url, params, headers):
        self._url = url
        self._params = params
        self._headers = headers
        self._session = None

    def _request(self, last_id=-1):
        if not self._session:
            self._session = requests.Session()
            self._session.headers = self._headers
            self._session.verify = '/etc/ssl/certs/ca-certificates.crt'
        self._session.params = self._get_params(last_id)

        response = self._session.get(self._url)
        if response.status_code != 200:
            raise StaffError("cannot fetch {} status {}".format(response.url, response.status_code))

        result = response.json()['result']
        log.debug("got {} results from {}".format(len(result), response.url))
        return result

    def get_all(self):
        last_id = -1
        results = []

        log.info("getting all staff entities from {}".format(self._url))
        page = self._request(last_id)
        while page:
            log.info("got {} results from staff".format(len(page)))
            log.debug("{}".format(str(page)))
            results.extend(page)
            last_id = page[-1]['id']
            page = self._request(last_id)

        return results

    def _get_params(self, last_id=-1):
        params = {}
        params.update(self._params)
        if '_fields' in params:
            params['_fields'] = ','.join(['id'] + params['_fields'].split(','))
        else:
            params['_fields'] = 'id'
        params['_sort'] = 'id'
        if '_query' in params:
            params['_query'] = ' and '.join([params['_query'], 'id>{}'.format(last_id)])
        else:
            params['_query'] = ['id>{}'.format(last_id)]
        params['_limit'] = 200
        return params
