class Version(object):
    def __init__(self, epoch, upstream_version, debian_revision):
        self.epoch = epoch
        self.upstream_version = upstream_version
        self.debian_revision = debian_revision

    @classmethod
    def from_string(cls, ver):
        epoch_end = ver.find(":")
        if epoch_end > 0:
            epoch = int(ver[:epoch_end])
            upstream_begin = epoch_end + 1
        else:
            epoch = 0
            upstream_begin = 0
        dr_begin = ver.rfind("-")
        if dr_begin > upstream_begin + 1:
            debian_revision = ver[dr_begin + 1:]
        else:
            debian_revision = "0"
            dr_begin = len(ver)
        upstream_version = ver[upstream_begin:dr_begin]
        return cls(epoch, upstream_version, debian_revision)

    # comparision code ported as-is from aptly/deb/version.go
    @staticmethod
    def _debian_lex(s1, s2):
        i = 0
        l1, l2 = len(s1), len(s2)
        while True:
            if i == l1 and i == l2:
                # s1 equal to s2
                break
            if i == l2:
                # s1 is longer
                if s1[i] == '~':
                    return -1  # s1 < s2
                return 1  # s1 > s2
            if i == l1:
                # s2 is longer
                if s2[i] == '~':
                    return 1  # s1 > s2
                return -1  # s1 < s2
            if s1[i] == s2[i]:
                i += 1
                continue
            if s1[i] == '~':
                return -1
            if s2[i] == '~':
                return 1
            c1, c2 = s1[i].isalpha(), s2[i].isalpha()
            if c1 and not c2:
                return -1
            if not c1 and c2:
                return 1
            if s1[i] < s2[i]:
                return -1
            return 1
        return 0

    @staticmethod
    def _atoi(s):
        try:
            return int(s)
        except ValueError:
            return 0

    @staticmethod
    def _cmp_part(part1, part2):
        i1, i2 = 0, 0
        l1, l2 = len(part1), len(part2)
        while True:

            j1, j2 = i1, i2
            while j1 < l1 and not part1[j1].isdigit():
                j1 += 1

            while j2 < l2 and not part2[j2].isdigit():
                j2 += 1
            s1, s2 = part1[i1:j1], part2[i2:j2]
            r = Version._debian_lex(s1, s2)
            if r != 0:
                return r
            i1, i2 = j1, j2

            while j1 < l1 and part1[j1].isdigit():
                j1 += 1

            while j2 < l2 and part2[j2].isdigit():
                j2 += 1

            s1, s2 = part1[i1:j1], part2[i2:j2]
            n1 = Version._atoi(s1)
            n2 = Version._atoi(s2)

            if n1 < n2:
                return -1

            if n1 > n2:
                return 1

            i1, i2 = j1, j2

            if i1 == l1 and i2 == l2:
                break
        return 0

    def _cmp(self, other):
        if self.epoch < other.epoch:
            return -1
        elif self.epoch > other.epoch:
            return 1
        r = Version._cmp_part(self.upstream_version, other.upstream_version)
        if r != 0:
            return r
        return Version._cmp_part(self.debian_revision, other.debian_revision)

    def __lt__(self, other):
        if self._cmp(other) < 0:
            return True
        else:
            return False

    def __gt__(self, other):
        if self._cmp(other) > 0:
            return True
        else:
            return False

    def __ge__(self, other):
        c_r = self._cmp(other)
        if c_r >= 0:
            return True
        else:
            return False

    def __le__(self, other):
        c_r = self._cmp(other)
        if c_r <= 0:
            return True
        else:
            return False

    def __eq__(self, other):
        c_r = self._cmp(other)
        if c_r == 0:
            return True
        else:
            return False

    def __ne__(self, other):
        c_r = self._cmp(other)
        if c_r != 0:
            return True
        else:
            return False

    def __str__(self):
        fmt = ""
        arg = []
        if self.epoch:
            fmt += "{}:"
            arg.append(self.epoch)
        fmt += "{}"
        arg.append(self.upstream_version)
        if self.debian_revision != "0":
            fmt += "-{}"
            arg.append(self.debian_revision)
        return fmt.format(*arg)

    def __repr__(self):
        return self.__str__()
