#!/usr/bin/env python

from __future__ import division, absolute_import
import sys
import os
import pwd
import grp

from infra.dist.dmover.lib.server import Server
from infra.dist.dmover.lib.context import Context
from infra.dist.dmover.lib.driver import Driver


def formatExc(msg, e):
    args = msg, e.__class__.__name__, " ".join(str(i) for i in e.args)
    return "{0}: type='{1}' args='{2}'".format(*args)


def main(argv):
    exitCode = 97
    ctx = None
    try:
        ctx = Context()
        ctx.boot(argv)

        ruid, euid, suid = os.getresuid()
        if suid != 0:
            ctx.reopenLogFromConfig()
            ctx.log.warning("I am not root, so it is not possible to drop privileges to another user! Operating under current user!")
        else:
            user_name = ctx.cfg['user']

            pwRec = pwd.getpwnam(ctx.cfg['user'])
            uid = pwRec.pw_uid
            gid = pwRec.pw_gid

            if 'group' in ctx.cfg:
                group = grp.getgrnam(ctx.cfg['group'])
                gid = group.gr_gid

            gids = set()
            for group in grp.getgrall():
                if pwRec.pw_name in group.gr_mem:
                    gids.add(group.gr_gid)
            gids.add(gid)

            os.setgroups(list(gids))
            os.setresgid(gid, gid, gid)
            os.setresuid(uid, uid, uid)

            ctx.reopenLogFromConfig()
            ctx.log.info("Dropped privileges to user %s (uid=%d, gid=%d, groups=(%s))" % (pwRec.pw_name, uid, gid, ", ".join([str(g_id) for g_id in gids])))

        try:
            from setproctitle import setproctitle
            setproctitle(ctx.cfg['programName'])
        except ImportError:
            pass

        driver = Driver(ctx)
        Server(ctx, driver).start().join()
    except SystemExit:
        raise
    except KeyboardInterrupt:
        ctx.log.info("exiting on user request")
        exitCode = 0
    except Exception as e:
        ctx.log.exception(formatExc("exiting on error", e))
        exitCode = 1
    sys.exit(exitCode)


if __name__ == '__main__':
    main(sys.argv)
