import sys
import logging
import logging.handlers
import argparse

from .config import getConfig, getDefaultConfig


class Context(object):
    def __init__(self):
        self._c_handler = logging.StreamHandler(sys.stdout)
        self._e_handler = logging.StreamHandler(sys.stderr)
        self._log_formatter = logging.Formatter('[%(asctime)s][%(name)s] - %(levelname)s - %(message)s')
        self._c_handler.setFormatter(self._log_formatter)
        self._e_handler.setFormatter(self._log_formatter)
        self.log = logging.getLogger('main')
        for h in self.log.handlers:
            self.log.removeHandler(h)
        self.tlog = logging.getLogger('timings')
        for h in self.tlog.handlers:
            self.tlog.removeHandler(h)
        self.log.addHandler(self._c_handler)
        self.log.setLevel(logging.DEBUG)
        self.tlog.addHandler(self._e_handler)
        self.tlog.setLevel(logging.DEBUG)
        self.cfg = None

    def boot(self, argv):
        self.args = self.parseArgs(argv[1:])
        if self.args.cfg:
            self.cfg = getConfig(self.args.cfg)
            self.log.info("using '{0}' as config".format(self.args.cfg))
        else:
            self.cfg = getDefaultConfig()
            self.log.info("using hardwired config")

    def reopenLogFromConfig(self):
        """
        If log settings are specified in the config, reopen log according to them
        Should be called after the privileges are dropped from root
        to the user specified in the config
        """
        if 'log' in self.cfg and 'path' in self.cfg['log']:
            if not self.args.console_log:
                try:
                    handler = logging.handlers.WatchedFileHandler(self.cfg['log']['path'])
                    handler.setFormatter(self._log_formatter)
                    self.log.addHandler(handler)
                    self.log.removeHandler(self._c_handler)
                    self.log.debug("Reopened log to %s" % self.cfg['log']['path'])
                except:
                    self.log.warning("Could not reopen log to %s" % self.cfg['log']['path'])

        if 'log' in self.cfg and 'root_log' in self.cfg['log']:
            if not self.args.console_log:
                try:
                    handler = logging.handlers.WatchedFileHandler(self.cfg['log']['path'])
                    handler.setFormatter(self._log_formatter)
                    logging.root.addHandler(handler)
                    logging.root.removeHandler(self._c_handler)
                    logging.debug("Reopened log to %s" % self.cfg['log']['path'])
                except:
                    self.log.warning("Could not reopen log to %s" % self.cfg['log']['path'])

        if 'log' in self.cfg and 'timings' in self.cfg['log']:
            try:
                handler = logging.handlers.WatchedFileHandler(self.cfg['log']['timings'])
                handler.setFormatter(self._log_formatter)
                self.tlog.addHandler(handler)
                self.tlog.removeHandler(self._e_handler)
                self.log.debug("Reopened timings log to %s" % self.cfg['log']['timings'])
            except:
                self.log.warning("Could not reopen timings log to %s" % self.cfg['log']['timings'])
        if 'log' in self.cfg and 'level' in self.cfg['log']:
            if hasattr(logging, self.cfg['log']['level']):
                self.log.setLevel(getattr(logging, self.cfg['log']['level']))
                logging.root.setLevel(getattr(logging, self.cfg['log']['level']))
            else:
                self.log.warning("Unknown log level %s" % self.cfg['log']['level'])

    def parseArgs(self, argv):
        parser = argparse.ArgumentParser(description='Conductor agent service')
        parser.add_argument('-c', '--cfg',
                            default=None,
                            action='store', help='path to service cfg file')
        parser.add_argument('--console-log',
                            action='store_true',
                            help='log to stdout')
        return parser.parse_args(argv)
