import gevent
import time
from collections import defaultdict
from gevent.lock import BoundedSemaphore
from .internal.repo import Repo
from .internal.exc import RetryableDmoveError, DmoveError
from .internal.locker import Locker
import kazoo.exceptions as kzexcept


class Driver(object):
    def __init__(self, context):
        """
        @type context: driver.context.Context
        """
        self.ctx = context
        self.ctx.dmove_semaphore = BoundedSemaphore(100)
        self.ctx.locker = Locker(context)

    def dmove(self, request):
        """
        @type request: infra.dist.dmover.lib.internal.dmove_request.DmoveRequest
        """
        expire_time = time.time() + int(self.ctx.cfg['repos']['dmove34Retry'])
        min_retry = int(self.ctx.cfg['repos']['dmove34MinRetryCount'])
        max_retry = int(self.ctx.cfg['repos']['dmove34MaxRetryCount'])
        retry_count = 0
        while time.time() < expire_time or retry_count <= min_retry:
            retry = False
            for task in self.prepare_tasks(request):
                try:
                    task.get()
                except RetryableDmoveError:
                    if retry_count >= max_retry:
                        raise
                    retry = True
                except DmoveError:
                    if retry_count >= max_retry:
                        raise
                    retry = True
                except kzexcept.KazooException:
                    if retry_count >= max_retry:
                        raise
                    retry = True
            if not retry:
                break
            retry_count += 1
            if retry_count > max_retry:
                break
            gevent.sleep(5)

    def prepare_tasks(self, request):
        dmove_tasks = []
        for repo in request.repos:
            packages = request.packs_by_repo(repo)
            dmove_tasks.append(gevent.spawn(self.do_repo_tasks, repo, packages, request.target_branch, request.force))

        return dmove_tasks

    def _filter_one_location_per_source(self, locations, sources):
        for packages in sources.itervalues():
            first_package = packages[0]
            for location in locations:
                if location.package == first_package:
                    yield location
                    break

    def do_repo_tasks(self, repo, packages, target_branch, force=False):
        tasks = []
        locations = {}

        # CONDUCTOR-645: acquire lock for the whole process of detecting
        # and moving packages to avoid race conditions when there are
        # other tasks dmoving package from the same source
        source_packages = defaultdict(list)

        _start = time.time()
        for package in packages:
            source_packages[package.source_package].append(package)
        self.ctx.tlog.info('Find source packages: %.4f' % (time.time() - _start))

        _start = time.time()
        all_locations = repo.locate_packages(packages)
        self.ctx.tlog.info('Locate packages: %.4f' % (time.time() - _start))

        _start = time.time()
        with self.ctx.locker.source_package_lock(source_packages.keys()):
            self.ctx.tlog.info('Lock wait: %.4f' % (time.time() - _start))
            self.ctx.log.info(
                'Took lock for source packages %s' % repr(
                    source_packages.keys())
            )
            # CONDUCTOR-645: Dmove only one of several packages that has one source package.
            # It should be enough to dmove one package per source for all packages to move
            raw_locations = self._filter_one_location_per_source(
                all_locations, source_packages)
            raw_locations = list(raw_locations)
            # raw_locations = repo.locate_packages([packs[0] for packs in source_packages.itervalues()])

            # ADMINTOOLS-493: we may be requested to dmove packages
            # before they appear in Packages file but after .deb and .changes
            # files are in place. For this case we suppose that missing
            # packages are from unstable branch and run dmove with that
            for loc in raw_locations:
                if loc.is_missing():
                    self.ctx.log.info('Location %s is missing, trying unstable' % repr(loc))
                    loc.branch = 'unstable'

            # CONDUCTOR-319: when we just dmoved a package the index in the
            # lower branch is not rebuilt yet, so we can have two locations
            # for one package. In this case we need to launch only one dmove
            # from the higher branch
            for loc in raw_locations:
                if loc.package in locations:
                    my_weight = Repo.BRANCHES_WEIGHTS[loc.branch]
                    other_weight = Repo.BRANCHES_WEIGHTS[locations[loc.package].branch]
                    if my_weight > other_weight:
                        locations[loc.package] = loc
                else:
                    locations[loc.package] = loc

            repo.dmove(
                locations.values(),
                target_branch,
                force,
                all_locations=all_locations
            )
        self.ctx.log.info('Released lock for source packages %s' % repr(source_packages.keys()))
