import logging
import os
import socket
import time

import kazoo.client as kzclient
import kazoo.exceptions as kzexcept

log = logging.getLogger(__name__)


def retry_once(what, *args, **kwargs):
    try:
        return what(*args, **kwargs)
    except Exception:
        log.exception("cannot call {} retrying once".format(what.__name__))
        return what(*args, **kwargs)


class ZKLockFactory(object):
    def __init__(self, logger=None, ctx=None):
        self.ctx = ctx
        self._connection = None
        self._init_connection()
        self._locks = {}

    def _init_connection(self):
        port = str(self.ctx.cfg['distributed_locks']['zk_port'])
        hosts = ','.join(
            ['{}:{}'.format(host, port)
             for host in self.ctx.cfg['distributed_locks']['zk_nodes']]
        )
        if self.ctx.cfg['distributed_locks']['supress_zk_log']:
            zk = kzclient.KazooClient(hosts=hosts)
        else:
            zk = kzclient.KazooClient(hosts=hosts, logger=self.ctx.log)
        zk.start()
        self._connection = zk

    def reconnect(self):
        try:
            self._connection.stop()
            self._connection.close()
            self._init_connection()
        except kzexcept.KazooException:
            log.exception("cannot reconnect to zookeeper")

    def get_lock(self, lock_id, raw=False):
        lock = None
        retry_count = 0
        retry_limit = self.ctx.cfg['distributed_locks']['zk_retry_limit']
        while retry_count < retry_limit:
            retry_count += 1
            try:
                lock = retry_once(self._connection.Lock, self._format_lock_path(lock_id), self._format_lock_id(lock_id))
                break
            except kzexcept.KazooException:
                self.ctx.log.exception('zookeeper connection problem retrying...')
                self.reconnect()
                if retry_count >= retry_limit:
                    raise
                else:
                    continue
        return lock

    def cleanup_lock(self, lock_id):
        retry_count = 0
        retry_limit = self.ctx.cfg['distributed_locks']['zk_retry_limit']
        while retry_count < retry_limit:
            retry_count += 1
            try:
                lock_path = self._format_lock_path(lock_id)
                try:
                    retry_once(self._connection.delete, lock_path)
                except kzexcept.NotEmptyError:
                    self.ctx.log.error('cannot clean up lock: %s, NotEmptyError', lock_id)
                break
            except kzexcept.KazooException:
                self.ctx.log.exception('zookeeper connection problem retrying...')
                self.reconnect()
                if retry_count >= retry_limit:
                    raise
                else:
                    continue

    def _format_lock_path(self, lock_id):
        return 'conductor-driver/{}/{}'.format(
            self.ctx.cfg['distributed_locks']['zk_prefix'],
            lock_id
        )

    def _format_lock_id(self, lock_id):
        return '{}_conductor-driver_{}_lock_id:{}_{}'.format(
            self.ctx.cfg['distributed_locks']['zk_prefix'],
            lock_id,
            socket.gethostname(),
            os.getpid()
        )


class Locker(object):
    class _SourcePackageLocker(object):

        def __init__(self, locker, source_packages):
            self.locker = locker
            self.source_packages = source_packages
            self._locks = dict()

        def __enter__(self):
            retry_count = 0
            while retry_count < 3:
                retry_count += 1
                try:
                    # locations are made unique and sorted before locking to avoid deadlocks
                    for source_package in sorted(set(self.source_packages)):
                        lock = self.locker.lock_provider.get_lock(source_package)
                        self._locks[source_package] = lock
                        retry_once(self._locks[source_package].acquire, timeout=30)
                    return self
                except kzexcept.KazooException:
                    log.exception("cannot take lock for {}".format(str(sorted(set(self.source_packages)))))
                    self.locker.lock_provider.reconnect()

                time.sleep(1)
            # sacrifice self
            self._sacrifice()

        def __exit__(self, exc_type, exc_val, exc_tb):
            retry_count = 0
            while retry_count < 3:
                retry_count += 1
                try:
                    for source_package in set(self.source_packages):
                        self._locks[source_package].release()
                        del self._locks[source_package]
                        self.locker.lock_provider.cleanup_lock(source_package)
                    return
                except Exception:
                    log.exception("cannot release lock for {}".format(str(sorted(set(self.source_packages)))))

                time.sleep(1)
            # sacrifice self
            self._sacrifice()

        @staticmethod
        def _sacrifice():
            log.error("cannot handle zookeeper error. should sacrifice daemon")
            # logging.shutdown()
            # pid = os.getpid()
            # os.kill(pid, 9)

    def __init__(self, ctx):
        self.lock_provider = ZKLockFactory(ctx=ctx)

    def source_package_lock(self, source_packages):
        return self._SourcePackageLocker(self, source_packages)
