from __future__ import division

import time
import gevent
import os
import re
import gzip
from gevent import subprocess
from urllib2 import urlopen, quote
from json import loads
from lxml import etree
from ConfigParser import ConfigParser
from .exc import DmoveError, RetryableDmoveError
from .package import Package


class Location(object):

    def __init__(self, package, repo, branch):
        self.package = package
        self.repo = repo
        self.branch = branch

    def is_missing(self):
        return self.branch is None

    def __hash__(self):
        return hash((
            self.package.name,
            self.package.version,
            self.repo.name,
            self.branch,
        ))

    def __eq__(self, other):
        if self.__hash__() == other.__hash__():
            return True
        return False

    def __str__(self):
        return '%s @ %s/%s' % (self.package, self.repo.name, self.branch)

    def __repr__(self):
        return self.__str__()


class DebianLocation(Location):

    def __init__(self, package, repo, branch, arch):
        self.arch = arch
        super(DebianLocation, self).__init__(package, repo, branch)

    def __str__(self):
        return super(DebianLocation, self).__str__() + ('(%s)' % self.arch)

    def __hash__(self):
        return hash((self.arch, super(DebianLocation, self).__hash__()))


class RedhatLocation(Location):

    def __init__(self, package, repo, branch, arch, package_arch):
        self.arch = arch
        self.package_arch = package_arch
        super(RedhatLocation, self).__init__(package, repo, branch)

    def __hash__(self):
        return hash((self.arch, self.package_arch, super(RedhatLocation, self).__hash__()))


def make_repo(name, info, context):
    if info['os_type'] == 'debian':
        return DebianRepo(name, info, context)

    elif info['os_type'] in ('redhat', 'redhat6', 'redhat6-unbranched'):
        if info['os_type'] == 'redhat':
            rh_version = 5
        else:
            rh_version = 6

        return RedhatRepo(name, info, context, rh_version)

    else:
        raise ValueError


class Repo(object):
    BRANCHES_WEIGHTS = dict(
        unstable=0,
        testing=10,
        prestable=20,
        stable=30
    )

#    @classmethod
#    def set_root_path(cls, path):
#        cls.root_path = path

    def __init__(self, name, info, context):
        self.branches = 'unstable', 'testing', 'prestable', 'stable'
        self.ctx = context
        self.name = name
        self.cfg = self.ctx.cfg['repos']
        self.root_path = self.cfg['path']
        self.log = self.ctx.log
        self.dmove_key = info['dmove_key']
        self.os_type = info['os_type']
        self.cacus = True

    def archs(self, branch):
        return []

    def grep(self, branch, arch, packages):
        raise NotImplementedError()

    def grep_cacus(self, packages):
        raise NotImplementedError()

    def repull(self, branch):
        raise NotImplementedError()

    def filter_requested_packages(self, packs):
        raise NotImplementedError()

    # def wait_packages(self, packages, branch):
    #     left_packages = packages[:]
    #     start = time.time()
    #     has_repulled = False
    #
    #     while left_packages:
    #         elapsed = time.time() - start
    #
    #         if elapsed > self.cfg['dmoveTimeout']:
    #             break
    #
    #         elif elapsed > (self.cfg['dmoveTimeout'] / 2) and not has_repulled:
    #
    #             try:
    #                 self.repull(branch)
    #             except NotImplementedError:  # Some repos may not have repull feature
    #                 pass
    #             has_repulled = True
    #
    #         gevent.sleep(1)
    #
    #         self.log.debug("Checking %s/%s for dmoved packages", self.name, branch)
    #
    #         for arch in self.archs(branch):
    #             found_locations = self.grep(branch, arch, left_packages)
    #             for loc in found_locations:
    #                 left_packages.remove(loc.package)
    #                 self.log.info("Found %s=%s in %s/%s", loc.package.name, loc.package.version, self.name, branch)
    #
    #     return left_packages

    def locate_packages(self, packages):
        """
        @type packages: list
        """
        locations = set()
        packages = packages[:]
        if self.cacus:
            locations.update(self.grep_cacus(packages))
        else:
            for branch in self.branches:
                for arch in self.archs(branch):
                    locations.update(self.grep(branch, arch, packages))

        found = set([loc.package for loc in locations])
        for missing_pack in set(packages) - found:
            locations.add(Location(missing_pack, self, None))

        return locations

    def detect_source_package(self, package):
        return package.name

    def invoke_dmove(self, dmove_key, to_branch, location, skip_reindex=False):
        """
        This is overrided by subclasses and also in tests
        """
        raise NotImplementedError()

    def _filter_locations(self, locations, target_branch, force):
        filtered_locations = []
        for loc in locations:
            if loc.branch is None:
                raise DmoveError("Cannot dmove from nowhere")
            correct_way = (
                self.BRANCHES_WEIGHTS[loc.branch] <
                self.BRANCHES_WEIGHTS[target_branch])
            force_condition = bool(force and loc.branch != target_branch)

            if correct_way or force_condition:
                filtered_locations.append(loc)
        return filtered_locations

    def _generate_reindexless_dmove_grns(self, locations, target_branch, force):
        dmove_grns = {}
        if not locations:
            return dmove_grns
        for loc in locations:
            dmove_grns[loc.package] = gevent.spawn(
                self.invoke_dmove,
                self.dmove_key,
                target_branch,
                loc,
                skip_reindex=True
            )
        return dmove_grns

    def dmove(self, locations, target_branch, force=False, all_locations=None):
        if len(self.branches) <= 1:
            return True
        dmove_grns = {}
        try:
            filtered_locations = self._filter_locations(
                locations, target_branch, force)
            dmove_grns = self._generate_reindexless_dmove_grns(
                filtered_locations, target_branch, force)

            for package, dmove_grn in dmove_grns.iteritems():
                _start = time.time()
                return_code, output = dmove_grn.get()

                self.ctx.tlog.info('Dmove: %.4f' % (time.time() - _start))
                if return_code == 0:
                    self.ctx.log.info(output)
                else:
                    self.ctx.log.error("Dmove command failed")
                    self.ctx.log.error(output)
                    raise DmoveError(
                        "Couldn't dmove %s=%s in %s. Dmove returned %d" % (
                            package.name,
                            package.version,
                            self.name,
                            return_code
                        )
                    )
            else:
                if all_locations:
                    self.repull(all_locations, target_branch)
                return True

        # We should wait for all processes before return.
        # Otherwise some locks may be released before dmove is finished
        finally:
            gevent.joinall(dmove_grns.values())

    def __hash__(self):
        return hash((self.name, self.root_path))

    def __eq__(self, other):
        return (self.name, self.root_path) == (other.name, other.root_path)

    def __ne__(self, other):
        return not self.__eq__(other)

    def __gt__(self, other):
        if self.root_path == other.root_path:
            return self.name > other.name
        else:
            return self.root_path > other.root_path

    def __lt__(self, other):
        if self.root_path == other.root_path:
            return self.name < other.name
        else:
            return self.root_path < other.root_path

    def __le__(self, other):
        return self.__lt__(other) or self.__eq__(other)

    def __ge__(self, other):
        return self.__gt__(other) or self.__eq__(other)


class RedhatRepo(Repo):
    def __init__(self, name, info, context, rh_version=6):
        super(RedhatRepo, self).__init__(name, info, context)
        self.rh_version = rh_version
        if self.os_type == 'redhat6-unbranched':
            self.branches = None,

    def archs(self, branch):
        return 'x86_64', 'i386'

    def get_path(self, branch, architecture):
        if self.os_type == 'redhat6-unbranched':
            return '%s/%s/%s/%s' % (
                self.root_path,
                self.name,
                self.rh_version,
                architecture
            )
        else:
            return '%s/%s/%s/%s/%s' % (
                self.root_path,
                self.name,
                branch,
                self.rh_version,
                architecture
            )

    def invoke_dmove(self, dmove_key, to_branch, location, skip_reindex=False):
        package_file = '%s-%s.%s.rpm' % (location.package.name, location.package.version, location.package_arch)
        command = (
            'sudo',
            'rhmove',
            to_branch,
            package_file,
            location.branch,
            location.arch,
            dmove_key,
        )
        with self.ctx.dmove_semaphore:
            self.ctx.log.info(command)
            proc = subprocess.Popen(command, stdout=subprocess.PIPE, stderr=subprocess.STDOUT)
            return proc.returncode, proc.communicate()[0]

    def grep(self, branch, architecture, packs):
        path = os.path.join(self.get_path(branch, architecture), 'repodata/other.xml.gz')

        # ACHTUNG: Tests are redefining python open
        # ACHTUNG: It is mandatory to use python open instead of opening file inside lxml
        try:
            other = etree.fromstring(gzip.GzipFile(fileobj=open(path)).read())
        except IOError as exc:
            if exc.errno == 2:
                return
            else:
                raise

        namespaces = {'other': 'http://linux.duke.edu/metadata/other'}
        for pack in packs:
            xpath = '/other:otherdata/other:package[@name="%s"]/other:version' % (pack.name,)
            for version in other.xpath(xpath, namespaces=namespaces):
                if pack.version == '%s-%s' % (version.attrib['ver'], version.attrib['rel']):
                    yield RedhatLocation(pack, self, branch, architecture, version.getparent().attrib['arch'])

    def repull(self, locations, target_branch):
        pass


class DebianRepo(Repo):

    def __init__(self, name, info, context):
        super(self.__class__, self).__init__(name, info, context)
        self.load_repo_cfg()

    def load_repo_cfg(self):
        configPathTemplate = self.ctx.cfg['repos']['configPathTemplate']
        cacusMarkerSection = self.ctx.cfg['repos']['cacusMarkerSection']
        configPath = configPathTemplate.format(self.name)
        if os.path.exists(configPath):
            iniparser = ConfigParser()
            iniparser.read(configPath)
            if iniparser.has_option(cacusMarkerSection, 'cacus'):
                if not iniparser.getboolean(cacusMarkerSection, 'cacus'):
                    self.cacus = False

    def get_path(self, branch, architecture):
        return '%s/%s/%s/%s' % (
            self.root_path,
            self.name,
            branch,
            architecture,
        )

    def archs(self, branch):
        debian_archs = ('all', 'amd64', 'i386')
        if self.root_path and self.name:
            path = '%s/%s/%s' % (self.root_path,
                                 self.name,
                                 branch)
            result = []
            for arch in debian_archs:
                if os.path.isdir(path + '/' + arch):
                    result.append(arch)
            return result
        else:
            return 'all', 'amd64', 'i386'

    def search_in_cacus(self, repo_name, pkg_name, pkg_ver):
        cacusSearchURLTemplate = self.ctx.cfg['repos']['cacusSearchURLTemplate']
        url_pattern = cacusSearchURLTemplate.format(repo_name)
        pkg_name = quote(pkg_name)
        pkg_ver = quote(pkg_ver)
        url = url_pattern.format(pkg_name, pkg_ver)
        response = urlopen(url).read()
        search_result = loads(response)
        return search_result

    # def grep_cacus(self, packs):
    #     for pack in packs:
    #         search_result = self.search_in_cacus(
    #             self.name, pack.name, pack.version)
    #         if search_result['success']:
    #             source_meta = search_result['result']
    #             if source_meta['version'] == pack.version:
    #                 for deb in source_meta['debs']:
    #                     if deb['package'] == pack.name:
    #                         yield DebianLocation(
    #                             pack,
    #                             self,
    #                             source_meta['environment'],
    #                             deb['architecture']
    #                         )
    #                         break

    def grep_cacus(self, packs):
        locations = set()
        for pack in packs:
            search_result = self.search_in_cacus(
                self.name, pack.name, pack.version)
            if search_result['success']:
                source_meta = search_result['result']
                for deb in source_meta['debs']:
                    locations.update([
                        DebianLocation(
                            pack,
                            self,
                            source_meta['environment'],
                            deb['architecture']
                        )
                    ])
        for loc in locations:
            yield loc

    def filter_requested_packages(self, packs, locations):
        for pack in packs:
            for loc in locations:
                if (
                    pack.name == loc.package.name and
                    pack.version == loc.package.version
                ):
                    yield loc
                    break

    def grep_local(self, branch, architecture, packs):
        path = os.path.join(self.get_path(branch, architecture), 'Packages')

        for p in open(path).read().split('\n\n'):
            name = None
            version = None
            for field in p.split('\n'):
                if field.startswith('Package:'):
                    name = field.split(':', 1)[1].strip()
                if field.startswith('Version:'):
                    version = field.split(':', 1)[1].strip()
            for pack in packs:
                if pack.name == name and pack.version == version:
                    yield DebianLocation(pack, self, branch, architecture)
                    break

    def grep(self, branch, architecture, packs):
        if self.name == 'turbo':
            raise NotImplementedError
        if self.cacus:
            locations = self.grep_cacus(packs)
            locations = self.filter_requested_packages(packs, locations)
            for location in locations:
                yield location
        else:
            for location in self.grep_local(branch, architecture, packs):
                yield location

    def get_paths(self):
        paths = []
        for branch in self.branches:
            for arch in self.archs(branch):
                paths.append((branch, arch))
        return paths

    def detect_source_package(self, package):
        if self.cacus:
            search_result = self.search_in_cacus(
                self.name, package.name, package.version)
            if search_result and search_result['success']:
                source_meta = search_result['result']
                return source_meta['source']
            return package.name
        devnull = open(os.devnull, 'w')
        for branch, arch in self.get_paths() + [('mini-dinstall', 'incoming')]:
            arch_repo_path = self.get_path(branch, arch)
            files = []

            for file_arch in self.archs(branch):
                version = re.sub('^.*?:', '', package.version)
                debfile = '%s_%s_%s.deb' % (package.name, version, file_arch)
                debfile = os.path.join(arch_repo_path, debfile)
                if os.path.exists(debfile):
                    files.append(debfile)

            if not files:
                continue

            cmd = ['dpkg', '-f']
            cmd += files
            cmd += ['--showformat', 'Source']
            self.ctx.log.info('Detecting source package with %s' % repr(cmd))
            proc = subprocess.Popen(cmd, stdout=subprocess.PIPE, stderr=subprocess.STDOUT)
            output = proc.communicate()[0]
            if proc.returncode == 0:
                for line in output.splitlines():
                    line = line.strip()
                    if line.startswith('Source:'):
                        return line.split()[1]
                self.ctx.log.warn('No source line in dpkg output, returning default')
                self.ctx.log.warn(output)
                return package.name
            else:
                self.ctx.log.warn('Error detecting source package')
                self.ctx.log.warn(output)
                return package.name
        self.ctx.log.warn('Failed detecting source package, returning default')
        raise RetryableDmoveError('Failed to find deb to detect source-package')

    def invoke_dmove(self, dmove_key, to_branch, location, skip_reindex=False):
        command = (
            'sudo',
            'dmove',
            dmove_key,
            to_branch,
            location.package.name,
            location.package.version,
            location.branch
        )
        if skip_reindex:
            command += ('--skipUpdateMeta=true',)
        # CONDUCTOR-645: retry dmove if dpkg -I failed and dmove returned 34 exit code
        #                This happens when two dmoves simultaneously launch with same
        #                source package and different debs
        with self.ctx.dmove_semaphore:
            self.ctx.log.info(command)
            proc = subprocess.Popen(command, stdout=subprocess.PIPE, stderr=subprocess.STDOUT)
            output = proc.communicate()[0]
            if proc.returncode != 34:
                if output.count('changes is absent') == 2:
                    self.ctx.log.info(output)
                    raise RetryableDmoveError()
                return proc.returncode, output
            else:
                self.ctx.log.info(output)
                raise RetryableDmoveError()

    # def repull(self, branch):
    #     repull_command = ('sudo', 'repull', self.name, branch)
    #     self.log.info("Repulling %s/%s", self.name, branch)
    #     ret_code = subprocess.call(repull_command)
    #     if ret_code != 0:
    #         raise DmoveError("Unable to repull %s/%s. Repull returned: %d" % (self.name, branch, ret_code))
    #     return True

    def _repull_atom(self, branch, arch):
        repull_command = (
            'sudo', 'cacus', 'update-repo', self.name, branch, arch)
        # repull_command = 'sudo cacus update-repo {} {} {}'.format(
        #    self.name, branch, arch)
        self.log.info(
            "Updating repo metadata for {}/{}/{}".format(
                self.name, branch, arch)
        )
        proc = subprocess.Popen(
            repull_command, stdout=subprocess.PIPE, stderr=subprocess.STDOUT)
        output = proc.communicate()[0]
        self.log.info(output)
        # ret_code, output = subprocess.getstatusoutput(repull_command)
        # ret_code = subprocess.call(repull_command)
        if proc.returncode != 0:
            self.log.error("Metadata update failed")
            self.log.error(output)
            raise RetryableDmoveError(
                "Unable to update metadata for repo {}/{}/{}. "
                "Return code: {}, output: {}".format(
                    self.name, branch, arch, proc.returncode, output)
            )

    def _prepare_unique_combinations(self, locations, target_branch):
        unique_locations = set()
        fake_package = Package('noname', '1', self)
        for loc in locations:
            src_loc = DebianLocation(
                fake_package, self, loc.branch, loc.arch)
            dst_loc = DebianLocation(
                fake_package, self, target_branch, loc.arch)
            unique_locations.update([src_loc, dst_loc])

        return list(unique_locations)

    def _skip_missing_locations(self, locations):
        for loc in locations:
            if not hasattr(loc, 'arch'):
                self.log.warn(
                    'Skipping metadata update for missing location: '
                    '{}/{}'.format(self.name, loc.branch)
                )

    def repull(self, locations, target_branch):
        self._skip_missing_locations(locations)
        filtered_locations = filter(lambda l: hasattr(l, 'arch'), locations)
        combinations = self._prepare_unique_combinations(
            filtered_locations, target_branch)
        for combination in combinations:
            self._repull_atom(combination.branch, combination.arch)
