from gevent import pywsgi, monkey, sleep
import json
import gevent.socket as socket
import gevent.event
import os
import traceback
from .internal.exc import DmoveError
from .internal.dmove_request import DmoveRequest

monkey.patch_all()


class LogProxy(object):
    def __init__(self, log):
        self.log = log

    def write(self, message):
        self.log.debug(message)


class Server(object):
    def __init__(self, context, driver):
        """
        @type context: driver.context.Context
        @type driver: driver.driver.Driver
        """
        self.ctx = context
        self.cfg = context.cfg
        self.driver = driver
        self._wsgi = None
        self._stopped = gevent.event.Event()
        self._stopped.set()
        self.slow_down = False

    def _sleep(self):
        if self.slow_down:
            sleep(1)

    def serve(self, env, start_response):
        self._sleep()
        if env['PATH_INFO'] == '/dmove' and env['REQUEST_METHOD'] == 'POST':
            try:
                jsondata = json.load(env['wsgi.input'])
                self.ctx.log.info("Serving request: %s", jsondata)
                data = DmoveRequest(self.ctx, jsondata)
            except (json.JSONDecodeError, ValueError):
                start_response('400 Bad Request', [('Content-Type', 'text/plain')])
                return ['JSON request could not be parsed or wrong arguments was passed']
            try:
                self.driver.dmove(data)
                start_response('200 OK', [('Content-Type', 'text/plain')])
                self.ctx.log.info("Request was successfully dmoved: %s", jsondata)
                return ['Dmoved']
            except DmoveError as exc:
                self.ctx.log.error("Encountered dmove error: %s", traceback.format_exc(exc))
                start_response('500 Internal Server Error', [('Content-Type', 'text/plain')])
                return ['Dmove error']
            except NotImplementedError:
                self.ctx.log.exception("Cannot dmove data (not implemented) '%s'" % jsondata)
                start_response('501 Not Implemented', [('Content-Type', 'text/plain')])
                return ['Not implemented error']
            except Exception as exc:
                self.ctx.log.exception("Unpredicted exception while dmoving: %s", traceback.format_exc(exc))
                start_response('500 Internal Server Error', [('Content-Type', 'text/plain')])
                return ['Unpredicted exception']
        elif env['PATH_INFO'] == '/ping':
            start_response('200 OK', [('Content-Type', 'text/plain')])
            return ['Ok']
        start_response('404 Not Found', [('Content-Type', 'text/plain')])
        return ['Requested path is not served']

    def start(self):
        self._stopped.clear()

        cfg = self.ctx.cfg['web']
        if cfg['mode'] == 'unix':
            socket_file = cfg['unixSocketPath']
            if os.access(socket_file, os.F_OK):
                if os.access(socket_file, os.W_OK):
                    os.unlink(socket_file)
                else:
                    raise RuntimeError("There's no write access to %s" % socket_file)

            # Socket should be writable by main user's group to communicate with web-server
            old_umask = os.umask(0002)
            listener = socket.socket(socket.AF_UNIX, socket.SOCK_STREAM)
            listener.bind(socket_file)
            listener.listen(10)
            os.umask(old_umask)
        elif cfg['mode'] == 'tcp':
            listener = ('', cfg['tcpPort'])
        else:
            raise RuntimeError("Unknown networking mode in config : '%s'" % cfg['mode'])

        self._wsgi = pywsgi.WSGIServer(
            listener,
            self.serve,
            log=LogProxy(self.ctx.log),
            environ={'SERVER_NAME': socket.getfqdn()}
        )
        self._wsgi.start()
        return self

    def stop(self):
        self._wsgi.stop()
        self._stopped.set()
        return self

    def join(self):
        self._stopped.wait()
