#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Provides: fs_free_space

import os


CHECKS = {
    'fs_free_space': {       # all significant host filesystems
        'errors': [],
        'mounts': ['/', '/logs', '/opt', '/repo']
    }
}

GB = 1024 * 1024 * 1024
MB = 1024 * 1024


def report():
    for check in sorted(CHECKS):
        if CHECKS[check]['errors']:
            print('PASSIVE-CHECK:{};2;Free space: {}'.format(check, ', '.join(CHECKS[check]['errors'])))
        else:
            print('PASSIVE-CHECK:{};0;Ok'.format(check))


def get_fs_stat(mountpoint):
    """
    Returns filesystem size and free space
    :param mountpoint: fs mountpoint path
    :return: (fs_size, free_size)
    """
    stat = os.statvfs(mountpoint)
    return (
        stat.f_blocks * stat.f_bsize,
        stat.f_bavail * stat.f_bsize
    )


def check():
    limits = {'big_size': 20 * GB, 'small_size': 500 * MB, 'percent': 5}

    mounts = {}
    for check in CHECKS:
        for point in CHECKS[check]['mounts']:
            if point in mounts:
                mounts[point].append(check)
            else:
                mounts[point] = [check]

    try:
        with open('/proc/mounts') as f:
            mounted = set(map(lambda x: x.split()[1], f.read().splitlines()))
    except Exception as e:
        for check in CHECKS:
            CHECKS[check]['errors'].append(str(e))
        return

    for mp in list(mounted.intersection(mounts.keys())):
        fs_size, fs_free = get_fs_stat(mp)
        free_perc = 100.0 * float(fs_free)/float(fs_size)
        if fs_size < 20 * GB and fs_free <= limits['small_size'] or \
           fs_size >= 20 * GB and fs_size < 50 * GB and free_perc <= limits['percent'] or \
           fs_size >= 50 * GB and fs_free <= limits['big_size']:
            error = '{} {}Gb({}%)'.format(mp, float(fs_free) / GB, free_perc)
            for check in mounts[mp]:
                CHECKS[check]['errors'].append(error)


if __name__ == '__main__':
    check()
    report()
