package cache

import (
	"bytes"
	"compress/gzip"
	"context"
	"fmt"
	"io/ioutil"

	"a.yandex-team.ru/infra/dist/repo-daemon/internal/cacus"
	"a.yandex-team.ru/infra/dist/repo-daemon/internal/mds"
)

const (
	Packages IndexType = iota
	Sources
	BundleIncompleteSizeThresold int = 200
)

type IndexType int

func (i IndexType) String() string {
	switch i {
	case Packages:
		return "Packages"
	case Sources:
		return "Sources"
	default:
		return "Unknown IndexType"
	}
}

type RepoIndexBundle struct {
	Release          []byte
	ReleaseGPG       []byte
	InRelease        []byte
	Plain            []byte
	GZIPed           []byte
	BZIP2ed          []byte
	ReleaseByHash    []byte
	ReleaseGPGByHash []byte
	InReleaseByHash  []byte
	BundleSize       int64
	IndexType        IndexType
	Incomplete       bool
	ByHash           bool
}

func (c *DistCache) FetchIndexBundle(ctx context.Context, index *cacus.Document) (*RepoIndexBundle, error) {
	var gzipped, bzipped, plain []byte
	var err error
	bundle := RepoIndexBundle{}
	bundle.Release = []byte(index.ReleaseFile)
	bundle.ReleaseGPG = []byte(index.ReleaseGpgContent)
	bundle.InRelease = []byte(index.InReleaseFileContent)
	bundle.ReleaseByHash = []byte(index.ReleaseByHashContent)
	bundle.ReleaseGPGByHash = []byte(index.ReleaseGPGByHashContent)
	bundle.InReleaseByHash = []byte(index.InReleaseByHashContent)
	if index.SourcesFilePath != "" {
		gzipped, err = c.mds.GetFileByKeyWithTimeout(ctx, index.GzippedSourcesPath, mds.RequestTimeout)
		if err != nil {
			return nil, fmt.Errorf("cannot get Sources.gz from %s: %s", index.GzippedSourcesPath, err)
		}
		bzipped, err = c.mds.GetFileByKeyWithTimeout(ctx, index.BzippedSourcesPath, mds.RequestTimeout)
		if err != nil {
			return nil, fmt.Errorf("cannot get Sources.bz2 from %s: %s", index.BzippedSourcesPath, err)
		}
		bundle.IndexType = Sources
	} else {
		gzipped, err = c.mds.GetFileByKeyWithTimeout(ctx, index.GzipedPackagesPath, mds.RequestTimeout)
		if err != nil {
			return nil, fmt.Errorf("cannot get Packages.gz from %s: %s", index.GzipedPackagesPath, err)
		}
		bzipped, err = c.mds.GetFileByKeyWithTimeout(ctx, index.BzipedPackagesPath, mds.RequestTimeout)
		if err != nil {
			return nil, fmt.Errorf("cannot get Packages.bz2 from %s: %s", index.BzipedPackagesPath, err)
		}
		bundle.IndexType = Packages
	}
	bundle.GZIPed = gzipped
	bundle.BZIP2ed = bzipped
	buffer := bytes.NewBuffer(bundle.GZIPed)
	gzReader, err := gzip.NewReader(buffer)
	if err != nil {
		return nil, fmt.Errorf("cannot create gzip reader for %s: %s", bundle.IndexType, err)
	}
	plain, err = ioutil.ReadAll(gzReader)
	if err != nil {
		return nil, fmt.Errorf("cannot gunzip %s: %s", bundle.IndexType, err)
	}
	bundle.Plain = plain
	//if we got less than BundleIncompleteSizeThresold bytes of data it is definitely an error and the bundle is incomplete
	if len(bundle.Plain) < BundleIncompleteSizeThresold || len(bundle.GZIPed) < BundleIncompleteSizeThresold || len(bundle.BZIP2ed) < BundleIncompleteSizeThresold {
		bundle.Incomplete = true
	}
	bs := int64(len(bundle.BZIP2ed))
	bs += int64(len(bundle.GZIPed))
	bs += int64(len(bundle.Plain))
	bs += int64(len(bundle.ReleaseGPG))
	bs += int64(len(bundle.InRelease))
	bs += int64(len(bundle.Release))
	bs += int64(len(bundle.InReleaseByHash))
	bs += int64(len(bundle.ReleaseGPGByHash))
	bs += int64(len(bundle.ReleaseByHash))
	bundle.BundleSize = bs
	bundle.ByHash = len(bundle.InReleaseByHash) > 0 && len(bundle.ReleaseGPGByHash) > 0 && len(bundle.ReleaseByHash) > 0
	return &bundle, nil
}

func (bundle *RepoIndexBundle) Size() int64 {
	return bundle.BundleSize
}
