package cache

import (
	"gopkg.in/yaml.v2"

	"a.yandex-team.ru/infra/dist/repo-daemon/pkg/fileutil"
	"a.yandex-team.ru/infra/dist/repo-daemon/pkg/logger"
)

const DefaultPath string = "cache.yaml"

type Param struct {
	Session struct {
		MaxSize int64 `yaml:"max_size"`
		// TTL in seconds
		TTL     int64  `yaml:"ttl"`
		Buckets uint32 `yaml:"buckets"`
		Prune   uint32 `yaml:"prune"`
	} `yaml:"session"`
	Data struct {
		MaxSize int64 `yaml:"max_size"`
		// TTL in seconds
		TTL     int64  `yaml:"ttl"`
		Buckets uint32 `yaml:"buckets"`
		Prune   uint32 `yaml:"prune"`
		// ByHashTTL in seconds
		ByHashTTL int64 `yaml:"by_hash_ttl"`
	} `yaml:"data"`
	Log logger.Config `yaml:"log"`
}

func DefaultParam() Param {
	p := Param{}
	p.Data.Buckets = 64
	p.Data.MaxSize = 4 * 1024 * 1024 * 1024
	p.Data.TTL = 10
	p.Data.ByHashTTL = 300
	p.Data.Prune = 128
	p.Session.TTL = 10
	p.Session.MaxSize = 100000
	p.Session.Buckets = 256
	p.Session.Prune = 5000
	return p
}

func LoadCacheYAML(path string) (Param, error) {
	var cacheYAML Param
	var filePath string
	cacheYAML = DefaultParam()
	switch {
	case fileutil.FileExists(path):
		filePath = path
	case fileutil.FileExists(DefaultPath):
		filePath = DefaultPath
	}
	rawYAML, err := fileutil.ReadFile(filePath)
	if err != nil {
		return cacheYAML, err
	}
	err = yaml.Unmarshal(rawYAML, &cacheYAML)
	if err != nil {
		return cacheYAML, err
	}
	return cacheYAML, nil
}
