package cacus

import (
	"context"
	"time"

	"go.mongodb.org/mongo-driver/bson"
	"go.mongodb.org/mongo-driver/bson/primitive"

	"a.yandex-team.ru/infra/dist/repo-daemon/pkg/logger"
)

type Document struct {
	ID                      primitive.ObjectID `bson:"_id"`
	Architecture            string             `bson:"architecture"`
	Environment             string             `bson:"environment"`
	Size                    int                `bson:"size"`
	PackagesFilePath        string             `bson:"packages_file"`
	GzipedPackagesPath      string             `bson:"gziped_packages"`
	PackagesSha1            primitive.Binary   `bson:"sha1"`
	ReleaseGpgContent       string             `bson:"release_gpg"`
	BzipedPackagesPath      string             `bson:"bziped_packages"`
	PackagesSha256          primitive.Binary   `bson:"sha256"`
	ReleaseFile             string             `bson:"release_file"`
	Lastupdated             time.Time          `bson:"lastupdated"`
	PackagesMd5             primitive.Binary   `bson:"md5"`
	DirtyBit                bool               `bson:"dirty_bit"`
	DirtyBitSetAt           time.Time          `bson:"dirty_bit_set_at"`
	InReleaseFileContent    string             `bson:"in_release_file"`
	ForceIndex              bool               `bson:"force_index"`
	SourcesFilePath         string             `bson:"sources_file"`
	GzippedSourcesPath      string             `bson:"gziped_sources"`
	BzippedSourcesPath      string             `bson:"bziped_sources"`
	ReleaseByHashContent    string             `bson:"release_by_hash"`
	ReleaseGPGByHashContent string             `bson:"release_gpg_by_hash"`
	InReleaseByHashContent  string             `bson:"in_release_gpg_by_hash"`
	PlainSHA256             string             `bson:"plain_sha256"`
	GzippedSHA256           string             `bson:"gzipped_sha256"`
	BzippedSHA256           string             `bson:"bzipped_sha256"`
}

type ReposIndexes map[string]*[]Document

func (db *DBClient) GetRepoIndices(ctx context.Context, repo string) (*[]Document, error) {
	if ctx == nil {
		ctx = context.Background()
	}
	c, err := db.GetCacusClient(ctx)
	if err != nil {
		return nil, err
	}
	col := db.GetCacusDB(c).Collection(repo)
	cur, err := col.Find(ctx, bson.M{})
	if err != nil {
		return nil, err
	}
	var indices []Document
	for cur.Next(ctx) {
		var result Document
		err := cur.Decode(&result)
		if err != nil {
			return nil, err
		}
		indices = append(indices, result)
	}
	return &indices, nil
}

func IndexDiffers(o, n *Document) bool {
	return o.InReleaseFileContent != n.InReleaseFileContent
}

func (db *DBClient) GetIndexWithTimeout(ctx context.Context, repo, env, arch string, t time.Duration) (*Document, error) {
	ctx, cancel := context.WithTimeout(ctx, t)
	defer cancel()
	return db.GetIndex(ctx, repo, env, arch)
}

func (db *DBClient) GetIndex(ctx context.Context, repo, env, arch string) (*Document, error) {
	if ctx == nil {
		ctx = context.Background()
	}
	logger.Debugf("%s/%s/%s: getting index from DB", repo, env, arch)
	result := new(Document)
	c, err := db.GetCacusClient(ctx)
	if err != nil {
		return nil, err
	}
	col := db.GetCacusDB(c).Collection(repo)
	one := col.FindOne(ctx, bson.M{"environment": env, "architecture": arch})
	err = one.Decode(result)
	if err != nil {
		return nil, err
	}
	return result, nil
}
