package cacus

import (
	"gopkg.in/yaml.v2"

	"a.yandex-team.ru/infra/dist/repo-daemon/pkg/fileutil"
)

const (
	DefaultPath                string = "/etc/cacus.yaml"
	DefaultMongoMaxConnections uint64 = 300
	DefaultMongoConnIdle       int    = 300
	DefaultPort                int    = 1488
)

type RepoDaemon struct {
	Port                int    `yaml:"port"`
	MongoMaxConnections uint64 `yaml:"mongo_max_connections"`
	MongoConnIdle       int    `yaml:"mongo_conn_idle"`
	EnforceSessions     bool   `yaml:"enforce_sessions"`
}

type StorageParam struct {
	AuthHeader     map[string]string `yaml:"auth_header"`
	ReadURL        string            `yaml:"read_url"`
	WriteURL       string            `yaml:"write_url"`
	TorrentURL     string            `yaml:"torrent_url"`
	ConnectTimeout int               `yaml:"connect_timeout"`
	ReadTimeout    int               `yaml:"read_timeout"`
}

type DBParam struct {
	Host       string `yaml:"host"`
	Port       int16  `yaml:"port"`
	DB         string `yaml:"db"`
	Username   string `yaml:"username"`
	Password   string `yaml:"password"`
	ReplicaSet string `yaml:"replicaset"`
}

type MetaDB struct {
	CacusDB DBParam `yaml:"cacusdb"`
	ReposDB DBParam `yaml:"reposdb"`
	HistDB  DBParam `yaml:"histdb"`
}

type ConfigYAML struct {
	RepoDaemon RepoDaemon   `yaml:"repo_daemon"`
	MetaDB     MetaDB       `yaml:"metadb"`
	Storage    StorageParam `yaml:"storage"`
}

func applyDefaults(daemon *RepoDaemon) {
	if daemon.Port == 0 {
		daemon.Port = DefaultPort
	}
	if daemon.MongoConnIdle == 0 {
		daemon.MongoConnIdle = DefaultMongoConnIdle
	}
	if daemon.MongoMaxConnections == 0 {
		daemon.MongoMaxConnections = DefaultMongoMaxConnections
	}
}

func LoadCacusYAML(path string) (ConfigYAML, error) {
	var cacusYAML ConfigYAML
	var filePath string
	switch {
	case fileutil.FileExists(path):
		filePath = path
	case fileutil.FileExists(DefaultPath):
		filePath = DefaultPath
	}
	rawYAML, err := fileutil.ReadFile(filePath)
	if err != nil {
		return cacusYAML, err
	}
	err = yaml.Unmarshal(rawYAML, &cacusYAML)
	if err != nil {
		return cacusYAML, err
	}
	applyDefaults(&cacusYAML.RepoDaemon)
	return cacusYAML, nil
}
