package cacus

import (
	"context"
	"fmt"
	"strings"
	"time"

	"go.mongodb.org/mongo-driver/mongo"
	"go.mongodb.org/mongo-driver/mongo/options"
)

const DBTimeout = 5 * time.Second

type DBClient struct {
	cacusMongoClient   *mongo.Client
	reposMongoClient   *mongo.Client
	historyMongoClient *mongo.Client
	cacusDBParam       *DBParam
	reposDBParam       *DBParam
	historyDBParam     *DBParam
	cacusURI           string
	reposURI           string
	historyURI         string
	maxConnIdle        int
	maxConn            uint64
	dbTimeout          time.Duration
}

func NewDBClient(cacus, repos, history DBParam, daemon RepoDaemon) *DBClient {
	db := DBClient{
		cacusMongoClient:   nil,
		reposMongoClient:   nil,
		historyMongoClient: nil,
		cacusDBParam:       &cacus,
		reposDBParam:       &repos,
		historyDBParam:     &history,
		maxConnIdle:        daemon.MongoConnIdle,
		maxConn:            daemon.MongoMaxConnections,
		dbTimeout:          DBTimeout,
	}
	db.cacusURI = db.cacusDBParam.String()
	db.reposURI = db.reposDBParam.String()
	db.historyURI = db.historyDBParam.String()
	return &db
}

func (dbp DBParam) String() string {

	split := strings.Split(dbp.Host, ",")
	hosts := make([]string, 0, len(split))
	for _, s := range split {
		hosts = append(hosts, fmt.Sprintf("%s:%d", s, dbp.Port))
	}
	hostPart := strings.Join(hosts, ",")
	return fmt.Sprintf(
		"mongodb://%s:%s@%s/%s?replicaSet=%s&readPreference=nearest",
		dbp.Username,
		dbp.Password,
		hostPart,
		dbp.DB,
		dbp.ReplicaSet)
}

func (db *DBClient) EstablishAllConnections(ctx context.Context) error {
	if ctx == nil {
		ctx = context.Background()
	}
	var err error
	db.cacusMongoClient, err = db.CreateMongoClient(db.cacusURI)
	if err != nil {
		return err
	}
	db.reposMongoClient, err = db.CreateMongoClient(db.reposURI)
	if err != nil {
		return err
	}
	db.historyMongoClient, err = db.CreateMongoClient(db.historyURI)
	if err != nil {
		return err
	}
	err = db.cacusMongoClient.Connect(ctx)
	if err != nil {
		return err
	}
	err = db.reposMongoClient.Connect(ctx)
	if err != nil {
		return nil
	}
	err = db.historyMongoClient.Connect(ctx)
	if err != nil {
		return err
	}
	return nil
}

func (db *DBClient) CreateMongoClient(uri string) (*mongo.Client, error) {
	return mongo.NewClient(
		options.Client().
			ApplyURI(uri).
			SetMaxConnIdleTime(time.Duration(db.maxConnIdle) * time.Second).
			SetMaxPoolSize(db.maxConn))
}

func (db *DBClient) GetCacusClient(ctx context.Context) (*mongo.Client, error) {
	if ctx == nil {
		ctx = context.Background()
	}
	err := db.cacusMongoClient.Ping(ctx, nil)
	if err != nil {
		return nil, err
	}
	return db.cacusMongoClient, nil
}

func (db *DBClient) GetReposClient(ctx context.Context) (*mongo.Client, error) {
	if ctx == nil {
		ctx = context.Background()
	}
	err := db.reposMongoClient.Ping(ctx, nil)
	if err != nil {
		return nil, err
	}
	return db.reposMongoClient, nil
}

func (db *DBClient) GetHistoryClient(ctx context.Context) (*mongo.Client, error) {
	if ctx == nil {
		ctx = context.Background()
	}
	err := db.historyMongoClient.Ping(ctx, nil)
	if err != nil {
		return nil, err
	}
	return db.historyMongoClient, nil
}

func (db *DBClient) GetCacusDB(m *mongo.Client) *mongo.Database {
	return m.Database(db.cacusDBParam.DB)
}

func (db *DBClient) GetReposDB(m *mongo.Client) *mongo.Database {
	return m.Database(db.reposDBParam.DB)
}

func (db *DBClient) GetHistoryDB(m *mongo.Client) *mongo.Database {
	return m.Database(db.historyDBParam.DB)
}
