package cacus

import (
	"context"
	"fmt"
	"strings"
	"time"

	"go.mongodb.org/mongo-driver/bson"
	"go.mongodb.org/mongo-driver/bson/primitive"
	"go.mongodb.org/mongo-driver/mongo/options"
)

type RepoDocument struct {
	ID          primitive.ObjectID `bson:"_id"`
	Sources     []SourceEntry      `bson:"sources"`
	AuditMeta   []AuditMeta        `bson:"audit_meta"`
	Environment string             `bson:"environment"`
	Source      string             `bson:"Source"`
	Version     string             `bson:"Version"`
	Debs        []DEBEntry         `bson:"debs"`
	Dsc         DSCEntry           `bson:"dsc"`
}
type SourceEntry struct {
	Size       int64            `bson:"size"`
	Sha1       primitive.Binary `bson:"sha1"`
	Name       string           `bson:"name"`
	StorageKey string           `bson:"storage_key"`
	Sha256     primitive.Binary `bson:"sha256"`
	Sha512     primitive.Binary `bson:"sha512"`
	Md5        primitive.Binary `bson:"md5"`
}
type AuditMeta struct {
	Timestamp time.Time `bson:"timestamp"`
	Event     string    `bson:"event"`
	User      string    `bson:"user"`
}
type DEBEntry struct {
	Size          int64            `bson:"size"`
	Priority      string           `bson:"Priority"`
	Sha1          primitive.Binary `bson:"sha1"`
	Maintainer    string           `bson:"Maintainer"`
	Description   string           `bson:"Description"`
	Package       string           `bson:"Package"`
	Homepage      string           `bson:"Homepage"`
	Section       string           `bson:"Section"`
	Depends       string           `bson:"Depends"`
	StorageKey    string           `bson:"storage_key"`
	Suggests      string           `bson:"Suggests"`
	InstalledSize string           `bson:"Installed-Size"`
	Version       string           `bson:"Version"`
	Architecture  string           `bson:"Architecture"`
	Sha256        primitive.Binary `bson:"sha256"`
	Sha512        primitive.Binary `bson:"sha512"`
	Md5           primitive.Binary `bson:"md5"`
	RBTorrentID   string           `bson:"rbtorrent_id"`
}
type DSCEntry struct {
	Binary           string `bson:"Binary"`
	Maintainer       string `bson:"Maintainer"`
	Format           string `bson:"Format"`
	BuildDepends     string `bson:"Build-Depends"`
	Source           string `bson:"Source"`
	Version          string `bson:"Version"`
	StandardsVersion string `bson:"Standards-Version"`
	Architecture     string `bson:"Architecture"`
	PackageList      string `bson:"Package-List"`
	VcsGit           string `bson:"Vcs-Git"`
	VcsBrowser       string `bson:"Vcs-Browser"`
	Homepage         string `bson:"Homepage"`
}

func (db *DBClient) GetRepoList(ctx context.Context) (*[]string, error) {
	if ctx == nil {
		ctx = context.Background()
	}
	c, err := db.GetReposClient(ctx)
	if err != nil {
		return nil, err
	}
	rdb := db.GetReposDB(c)
	listOptions := options.ListCollections()
	listOptions.SetNameOnly(true)
	cur, err := rdb.ListCollections(ctx, bson.M{}, listOptions)
	if err != nil {
		return nil, err
	}
	var repoList []string
	for cur.Next(ctx) {
		var result bson.M
		var repoName string
		err := cur.Decode(&result)
		if err != nil {
			return nil, err
		}
		repoName = result["name"].(string)
		if !strings.HasPrefix(repoName, "__") && repoName != "locks" {
			repoList = append(repoList, repoName)
		}
	}
	return &repoList, nil
}

func (db *DBClient) FindPackages(ctx context.Context, repo string, selector bson.M) (*[]RepoDocument, error) {
	if ctx == nil {
		ctx = context.Background()
	}
	c, err := db.GetReposClient(ctx)
	if err != nil {
		return nil, err
	}
	rdb := db.GetReposDB(c)
	col := rdb.Collection(repo)
	cur, err := col.Find(ctx, selector)
	if err != nil {
		return nil, err
	}
	result := make([]RepoDocument, 0)
	for cur.Next(ctx) {
		doc := RepoDocument{}
		err := cur.Decode(&doc)
		if err != nil {
			return nil, err
		}
		result = append(result, doc)
	}
	return &result, nil
}

func (db *DBClient) FindPackageByStorageKey(ctx context.Context, repo string, storageKey string) (*RepoDocument, error) {
	if ctx == nil {
		ctx = context.Background()
	}
	c, err := db.GetReposClient(ctx)
	if err != nil {
		return nil, err
	}
	rdb := db.GetReposDB(c)
	col := rdb.Collection(repo)
	result := RepoDocument{}
	err = col.FindOne(ctx, bson.M{"debs.storage_key": storageKey}).Decode(&result)
	if err != nil {
		return nil, err
	}
	return &result, nil
}

func (db *DBClient) FindSourceKeyBySourceFile(ctx context.Context, repo, env, file string) (string, error) {
	if ctx == nil {
		ctx = context.Background()
	}
	c, err := db.GetReposClient(ctx)
	if err != nil {
		return "", err
	}
	rdb := db.GetReposDB(c)
	col := rdb.Collection(repo)
	doc := RepoDocument{}
	err = col.FindOne(ctx, bson.M{"environment": env, "sources.name": file}).Decode(&doc)
	if err != nil {
		return "", err
	}
	for i := range doc.Sources {
		if doc.Sources[i].Name == file {
			return doc.Sources[i].StorageKey, nil
		}
	}
	return "", fmt.Errorf("could not find the file: %s in %s/%s sources", file, repo, env)
}
