package mds

import (
	"context"
	"fmt"
	"io"
	"io/ioutil"
	"net/http"
	"strings"
	"time"

	"a.yandex-team.ru/infra/dist/repo-daemon/internal/cacus"
	"a.yandex-team.ru/infra/dist/repo-daemon/pkg/logger"
)

const (
	distMDSNamespace string = "repo"
	RequestTimeout          = 20 * time.Second
)

type Client struct {
	storageParam cacus.StorageParam
	namespace    string
}

func NewMDSClient(param cacus.StorageParam) *Client {
	return &Client{
		storageParam: param,
		namespace:    distMDSNamespace,
	}

}

func (c *Client) GetFileByKeyWithTimeout(ctx context.Context, key string, t time.Duration) ([]byte, error) {
	ctx, cancel := context.WithTimeout(ctx, t)
	defer cancel()
	return c.GetFileByKey(ctx, key)
}

func (c *Client) GetFileByKey(ctx context.Context, key string) ([]byte, error) {
	var readURL string
	if key == "<empty>" {
		return make([]byte, 0), nil
	}
	if c.storageParam.ReadURL[len(c.storageParam.ReadURL)-1] == '/' {
		readURL = strings.TrimRight(readURL, "/")
	} else {
		readURL = c.storageParam.ReadURL
	}
	requestURL := fmt.Sprintf("%s/get-%s/%s", readURL, c.namespace, key)
	req, err := http.NewRequestWithContext(ctx, "GET", requestURL, nil)
	if err != nil {
		return nil, fmt.Errorf("cannot create request for key %s: %s", key, err)
	}
	logger.Infof("Fetching GET %s", requestURL)
	resp, err := http.DefaultClient.Do(req)
	if err != nil {
		return nil, fmt.Errorf("cannot fetch key %s: %s", key, err)
	} else {
		defer closeBody(resp.Body)
	}
	if resp.StatusCode != 200 {
		return nil, fmt.Errorf("got status: %d for request: %s", resp.StatusCode, requestURL)
	}
	return ioutil.ReadAll(resp.Body)
}

func closeBody(c io.Closer) {
	err := c.Close()
	if err != nil {
		logger.Errorf("Failed to close response body: %s", err)
	}
}
