package server

import (
	"encoding/json"
	"net/http"

	"github.com/go-chi/chi/v5"

	"a.yandex-team.ru/infra/dist/repo-daemon/internal/cacus"
	"a.yandex-team.ru/infra/dist/repo-daemon/pkg/logger"
)

type reposJSONEntry struct {
	Branches []string `json:"branches"`
	Type     string   `json:"type"`
	Name     string   `json:"name"`
	Dmove    string   `json:"dmove"`
}

type reposJSON []reposJSONEntry

type APIHandler struct {
	db *cacus.DBClient
}

func NewAPIHandler(db *cacus.DBClient) *APIHandler {
	return &APIHandler{db: db}
}

func (h *APIHandler) APIV1Router() chi.Router {
	router := chi.NewRouter()

	router.Get("/repos.json", h.reposJSONHandler)
	router.Get("/search", h.searchHandler)

	return router
}

func (h *APIHandler) reposJSONHandler(w http.ResponseWriter, r *http.Request) {
	ctx := r.Context()
	repoList, err := h.db.GetRepoList(ctx)
	if err != nil {
		logger.Errorf("repos.json: [%s] error getting repo list: %s", r.RemoteAddr, err)
		w.WriteHeader(500)
	}
	jsons := reposJSON{}
	for _, repo := range *repoList {
		indices, err := h.db.GetRepoIndices(ctx, repo)
		if err != nil {
			logger.Errorf("repos.json: [%s] error getting repo: %s index list: %s", r.RemoteAddr, repo, err)
			w.WriteHeader(500)
		}
		branchesSet := make(map[string]bool)
		jsonEntry := reposJSONEntry{
			Branches: []string{},
			Dmove:    repo,
			Name:     repo,
			Type:     "debian",
		}
		for _, index := range *indices {
			if !branchesSet[index.Environment] {
				branchesSet[index.Environment] = true
				jsonEntry.Branches = append(jsonEntry.Branches, index.Environment)
			}
		}
		jsons = append(jsons, jsonEntry)
	}
	JSON, err := json.Marshal(jsons)
	if err != nil {
		logger.Errorf("repos.json: [%s] error formatting json: %s", r.RemoteAddr, err)
		w.WriteHeader(500)
	}
	n, err := w.Write(JSON)
	if err != nil {
		logger.Errorf("Request GET %s interrupted after %d bytes written: %s", r.RequestURI, n, err)
		return
	}
}
