package server

import (
	"context"
	"fmt"
	"net/http"
	"time"

	"github.com/go-chi/chi/v5"

	"a.yandex-team.ru/infra/dist/repo-daemon/internal/cache"
	"a.yandex-team.ru/infra/dist/repo-daemon/internal/cacus"
	"a.yandex-team.ru/infra/dist/repo-daemon/pkg/logger"
)

type ByHashHandler struct {
	enforceSessions bool
	db              *cacus.DBClient
	cache           *cache.DistCache
}

func NewByHashHandler(enforceSessions bool, db *cacus.DBClient, c *cache.DistCache) ByHashHandler {
	return ByHashHandler{enforceSessions: enforceSessions, db: db, cache: c}
}

func (h *ByHashHandler) Handle(w http.ResponseWriter, r *http.Request) {
	hash := chi.URLParam(r, "hash")
	arch := chi.URLParam(r, "arch")
	repo := chi.URLParam(r, "repo")
	env := chi.URLParam(r, "env")
	ctx := r.Context()
	if h.enforceSessions {
		sessionItem, err := h.cache.GetSessionItem(ctx, repo, env, arch, r.RemoteAddr)
		if err != nil {
			logger.Error(err)
			w.WriteHeader(404)
			return
		}
		sessionData := sessionItem.Value().(*cache.SessionItem)
		if sessionData.DataItem.Value().(*cache.DataItem).Bundle.Incomplete {
			w.Header().Set("X-Request-State", "Incomplete data served to silence APT errors")
			logger.Debugf("%s[%s]: served incomplete request!", r.RequestURI, r.RemoteAddr)
		}
		err = h.writeResponse(ctx, w, r, sessionData, repo, env, arch, hash)
		if err != nil {
			logger.Errorf("/%s/%s/%s/%s: %s", repo, env, arch, hash, err)
		}
		readyFlag := h.cache.SetSessionCleanup(repo, env, arch, r.RemoteAddr)
		if sessionData.RequestBitmap&cache.BundleFetched == cache.BundleFetched || err != nil {
			readyFlag <- true
		}
		return
	} else {
		session := dummySession{}
		_ = h.writeResponse(ctx, w, r, &session, repo, env, arch, hash)
		return
	}
}

func (h *ByHashHandler) writeResponse(ctx context.Context, w http.ResponseWriter, r *http.Request, sessionData Session, repo, env, arch, hash string) error {
	code, data, err := h.fetchResponseData(ctx, w, r, repo, env, arch, hash)
	if err != nil {
		logger.Errorf("/%s/%s/%s/%s: %s", repo, env, arch, hash, err)
	}
	w.WriteHeader(code)
	if data != nil {
		n, err := w.Write(data)
		return fmt.Errorf("failed to write after %d bytes: %s", n, err)
	}
	sessionData.SetBitmap(sessionData.Bitmap() | cache.BundleFetched)
	return nil
}

func (h *ByHashHandler) fetchResponseData(ctx context.Context, w http.ResponseWriter, r *http.Request, repo, env, arch, hash string) (int, []byte, error) {
	var ifModifiedSince time.Time
	var err error
	var data *cache.ByHashItem
	if r.Header.Get("If-Modified-Since") != "" {
		ifModifiedSince, err = http.ParseTime(r.Header.Get("If-Modified-Since"))
		if err != nil {
			logger.Debugf("%s[%s]: cannot parse time string '%s': %s", r.RequestURI, r.RemoteAddr,
				r.Header.Get("If-Modified-Since"), err)
		}
	}
	data = h.cache.GetByHash(repo, env, arch, hash)
	if data == nil {
		currentIdx, err := h.db.GetIndexWithTimeout(ctx, repo, env, arch, cacus.DBTimeout)
		if err == nil {
			// trigger update if we have stale current data
			if hash == currentIdx.BzippedSHA256 || hash == currentIdx.GzippedSHA256 || hash == currentIdx.PlainSHA256 {
				_, err := h.cache.GetDataBundle(ctx, repo, env, arch, true)
				if err != nil {
					return 404, nil, err
				}
				data = h.cache.GetByHash(repo, env, arch, hash)
				if data == nil {
					return 404, nil, fmt.Errorf("cannot get hash %s from cache", hash)
				}
			} else {
				// look up history db for hash
				entry, err := h.db.GetHistoryEntryWithTimeout(ctx, repo, env, arch, hash, cacus.DBTimeout)
				if err != nil {
					return 404, nil, fmt.Errorf("hash %s not found in db: %s", hash, err)
				}
				if entry.ValidBefore.Before(time.Now()) {
					return 404, nil, fmt.Errorf("hash is valid before %s", entry.ValidBefore.String())
				}
				// locks managed by cache
				data, err = h.cache.FetchByHash(ctx, repo, entry)
				if err != nil {
					return 404, nil, err
				}
			}
		} else {
			return 404, nil, fmt.Errorf("hash %s is not in cache", hash)
		}
	}
	// check if data already invalid
	if !data.ValidBefore.IsZero() && data.ValidBefore.Before(time.Now()) {
		return 404, nil, fmt.Errorf("data for hash is valid before: %s", data.ValidBefore.String())
	}
	lastModified := data.UpdatedAt.Truncate(time.Second)
	w.Header().Set("Last-Modified", lastModified.Format(http.TimeFormat))
	if lastModified.Equal(ifModifiedSince) || lastModified.Before(ifModifiedSince) {
		return 304, nil, nil
	}
	return 200, data.Data, nil
}
